import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoUserAttribute } from 'amazon-cognito-identity-js';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface RegistrationControllerReq {
    email: string;
    password: string;
}

export async function registrationController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<RegistrationControllerReq>(req);
    const email = data.email;
    const password = data.password;

    if (!email || !password) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
                password: !password ? 'The field must be filled out' : undefined,
            },
        });
    }

    const attributes = [
        new CognitoUserAttribute({
            Name: 'email',
            Value: email,
        }),
    ];

    return new Promise((resolve) => {
        userPull.signUp(email, password, attributes, [], function (error, result) {
            if (error) {
                sendResJson(res, HttpStatusCode.BAD_REQUEST, {
                    code: HttpStatusCode.BAD_REQUEST,
                    error: {
                        password: error.message,
                    },
                });

                return resolve();
            }

            sendResJson(res, HttpStatusCode.OK, {
                code: HttpStatusCode.OK,
                data: result,
            });

            return resolve();
        });
    });
}
