import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoUser } from 'amazon-cognito-identity-js';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface RestoreControllerReq {
    email: string;
}

export async function restoreController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<RestoreControllerReq>(req);
    const email = data.email;

    if (!email) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
            },
        });
    }

    const cognitoUser = new CognitoUser({
        Username: email,
        Pool: userPull,
    });

    return new Promise((resolve) => {
        cognitoUser.forgotPassword({
            onSuccess(data) {
                sendResJson(res, HttpStatusCode.OK, {
                    code: HttpStatusCode.OK,
                    data,
                });

                resolve();
            },
            inputVerificationCode(data) {
                sendResJson(res, HttpStatusCode.OK, {
                    code: HttpStatusCode.OK,
                    data,
                });

                resolve();
            },
            onFailure(error) {
                sendResJson(res, HttpStatusCode.BAD_REQUEST, {
                    code: HttpStatusCode.BAD_REQUEST,
                    error: {
                        email: error.message,
                    },
                });

                resolve();
            },
        });
    });
}
