import { createServer, IncomingMessage, ServerResponse } from 'node:http';
import { performance } from 'node:perf_hooks';

import { authPageController } from 'controllers/authPageController';
import { confirmRegistrationController } from 'controllers/confirmRegistationController';
import { confirmRestoreController } from 'controllers/confirmRestoreController';
import { csrfController } from 'controllers/csrfController';
import { error404PageController } from 'controllers/error404PageController';
import { error500PageController } from 'controllers/error500PageController';
import { processCsrfToken } from 'controllers/helpers/processCsrfToken';
import { processIdToken } from 'controllers/helpers/processIdToken';
import { loginController } from 'controllers/loginController';
import { logoutController } from 'controllers/logoutController';
import { passwordResetController } from 'controllers/passwordResetController';
import { profilePageController } from 'controllers/profilePageController';
import { refreshController } from 'controllers/refreshController';
import { registrationController } from 'controllers/registrationController';
import { registrationPageController } from 'controllers/registrationPageController';
import { resendConfirmationCodeController } from 'controllers/resendConfirmationCodeController';
import { restoreController } from 'controllers/restoreController';
import { restorePageController } from 'controllers/restorePageController';
import { userController } from 'controllers/userController';
import { versionController } from 'controllers/versionController';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { code2level } from 'lib/logger/code2level';
import { logger } from 'lib/logger/logger';
import { getReqHost } from 'lib/request/getReqHost/getReqHost';
import { getReqId } from 'lib/request/getReqId/getReqId';
import { getReqURL } from 'lib/request/getReqURL/getReqURL';
import { sendResHtml } from 'lib/response/sendResHtml/sendResHtml';
import { sendResRedirect } from 'lib/response/sendResRedirect/sendResRedirect';
import { setResNoCacheHeaders } from 'lib/response/setResNoCacheHeaders/setResNoCacheHeaders';

const DEFAULT_PORT = 9000;

const DM_PORT = process.env.DM_PORT || DEFAULT_PORT;

async function requestHandler(
    req: IncomingMessage,
    res: ServerResponse,
    options: RequestHandlerOptions,
): Promise<void> {
    const method = req.method;

    // favicon fix
    if (req.url === '/favicon.ico') {
        return sendResHtml(res, HttpStatusCode.NOT_FOUND, 'HTTP404: Not Found');
    }

    // middlewares
    if (method !== 'GET') {
        const done = await processCsrfToken(req, res, options);

        if (done) {
            return;
        }
    }

    const { pathname, searchParams } = getReqURL(req);

    // middlewares /auth/api/*
    if (pathname.startsWith('/auth/api/')) {
        setResNoCacheHeaders(res);
    }

    // POST /auth/api/*
    if (method === 'POST') {
        if (pathname === '/auth/api/login') {
            return loginController(req, res, options);
        }

        if (pathname === '/auth/api/logout') {
            return logoutController(req, res, options);
        }

        if (pathname === '/auth/api/password/reset') {
            return passwordResetController(req, res, options);
        }

        if (pathname === '/auth/api/registration') {
            return registrationController(req, res, options);
        }

        if (pathname === '/auth/api/registration/confirm/resend') {
            return resendConfirmationCodeController(req, res, options);
        }

        if (pathname === '/auth/api/registration/confirm') {
            return confirmRegistrationController(req, res, options);
        }

        if (pathname === '/auth/api/restore') {
            return restoreController(req, res, options);
        }

        if (pathname === '/auth/api/restore/confirm') {
            return confirmRestoreController(req, res, options);
        }
    }

    // GET /version.json
    // GET /auth/api/*
    if (method === 'GET') {
        if (pathname === '/version.json') {
            return versionController(req, res, options);
        }

        if (pathname === '/auth/api/csrf') {
            return csrfController(req, res, options);
        }

        if (pathname === '/auth/api/user') {
            return userController(req, res, options);
        }

        if (pathname === '/auth/api/refresh') {
            return refreshController(req, res, options);
        }
    }

    // GET html pages
    if (method === 'GET') {
        const currentUser = await processIdToken(req, res, { logger });

        if (pathname === '/auth/profile') {
            return profilePageController(currentUser, req, res, options);
        }

        if (pathname.startsWith('/auth') && currentUser) {
            // accept only relative retpath params to prevent open redirect vulnerability
            const retpathValue = searchParams.get('retpath');
            const retpath =
                retpathValue && (retpathValue === '/' || /^\/[a-z0-9]/i.test(retpathValue))
                    ? retpathValue
                    : '/auth/profile';

            return sendResRedirect(res, retpath);
        }

        if (pathname === '/auth') {
            return authPageController(req, res, options);
        }

        if (pathname === '/auth/registration') {
            return registrationPageController(req, res, options);
        }

        if (pathname === '/auth/restore') {
            return restorePageController(req, res, options);
        }
    }

    return error404PageController(req, res, options);
}

const server = createServer((req, res) => {
    const startTime = performance.now();
    const reqId = getReqId(req) || '-';

    const requestLogger = logger.child({
        reqId,
        host: getReqHost(req),
    });

    requestLogger.info('[Req] %s %s', req.method || 'UNKNOWN', req.url);

    requestHandler(req, res, { logger: requestLogger })
        .catch((error) => {
            requestLogger.error(error, 'Request handler error');

            return error500PageController(req, res, { logger: requestLogger });
        })
        .catch((error) => {
            requestLogger.error(error, 'Impossible request handler error');
        })
        .finally(() => {
            requestLogger[code2level(res.statusCode)](
                '[Res] %s %s %d %sms',
                req.method || 'UNKNOWN',
                req.url,
                res.statusCode,
                performance.now() - startTime,
            );
        });
});

server.listen(DM_PORT, () => {
    logger.info(`Server running at http://localhost:${DM_PORT}/`);
});
