const path = require('path');
const webpack = require('webpack');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const MiniCssExtractPlugin = require("mini-css-extract-plugin");
const ForkTsCheckerWebpackPlugin = require('fork-ts-checker-webpack-plugin');
const vendorCacheGroups = require('./webpack.cache_groups');

const rootDir = '../admin_next/';
const tsconfig = path.resolve(rootDir, 'tsconfig.json');

module.exports = (mode) => ({
    mode,
    entry: {
        main: path.resolve(rootDir, 'src/core/components/App/index.tsx')
    },
    output: {
        path: path.resolve(rootDir, 'dist'),
        publicPath: '/',
        chunkFilename: '[name].[contenthash].js',
        filename: '[name].[contenthash].js',
    },
    resolve: {
        extensions: [".ts", ".tsx", ".js"],
        alias: {
            'react-virtualized/List': 'react-virtualized/dist/es/List',
            './dist/cpexcel.js': false,
            'xlsx': 'xlsx/dist/xlsx.mini.min.js'
        },
    },
    plugins: [
        new webpack.ContextReplacementPlugin(/moment[/\\]locale$/, /ru/),
        new ForkTsCheckerWebpackPlugin({
            async: false,
            issue: {
                scope: 'all',
            },
            typescript: {
                enabled: true,
                memoryLimit: 4096,
                configFile: tsconfig,
            }
        }),
        new MiniCssExtractPlugin({
            filename: '[name].[contenthash].css',
            chunkFilename: '[name].[contenthash].css',
            ignoreOrder: true,
        }),
        new HtmlWebpackPlugin({
            template: path.resolve(rootDir, './src/index.html'),
            filename: 'index.html'
        }),
        new webpack.DefinePlugin({
            'process.env.VERSION': JSON.stringify(process.env.VERSION),
            'process.env.RUNNING_MODE': JSON.stringify(process.env.RUNNING_MODE),
            'process.env.DISABLE_ORIGIN_CHECK': JSON.stringify(process.env.DISABLE_ORIGIN_CHECK)
        })
    ],
    module: {
        rules: [
            {
                test: /\.tsx?$/,
                exclude: /node_modules/,
                use: [
                    {
                        loader: `ts-loader`,
                        options: {
                            transpileOnly: true,
                            configFile: tsconfig,
                        }
                    }
                ]
            },
            {
                test: /react-datetime\.css$/,
                use: [
                    {
                        loader: 'style-loader',
                    },
                    {
                        loader: 'css-loader',
                        options: {
                            importLoaders: 1,
                            esModule: false,
                        }
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            postcssOptions: {
                                config: path.resolve(rootDir),
                            }
                        }
                    }
                ],
            },
            {
                test: /\.css$/,
                exclude: [/react-datetime\.css$/],
                use: [
                    {
                        loader: MiniCssExtractPlugin.loader,
                        options: {
                            esModule: false,
                        }
                    },
                    {
                        loader: 'css-loader',
                        options: {
                            modules: {
                                localIdentName: '[folder]__[local]___[hash:base64:5]',
                            },
                            importLoaders: 1,
                            esModule: false,
                        },
                    },
                    {
                        loader: 'postcss-loader',
                        options: {
                            postcssOptions: {
                                config: path.resolve(rootDir),
                            }
                        }
                    }
                ]
            },
            {
                test: /images(.)+\.svg$/,
                type: "asset",
                generator: {
                    filename: 'assets/[contenthash][ext]'
                },
            },
            {
                test: /\.(png|jpg|gif)$/,
                type: "asset",
                generator: {
                    filename: 'assets/[contenthash][ext]'
                },
            },
            {
                test: /inline-svg(.)+\.svg$/,
                use: 'svg-inline-loader',
            },
            // {
            //     test: /\.mp3$/,
            //     use: 'file-loader?name=sounds/[name].[ext]'
            // },
            {
                test: /fonts(.)+\.(eot|woff2|woff|ttf|svg)$/,
                type: "asset/resource",
                generator: {
                    filename: 'fonts/[contenthash][ext]'
                },
            },
            {
                test: /\.component\.svg$/,
                use: ['@svgr/webpack'],
            }
        ]
    },
    optimization: {
        splitChunks: {
            minSize: 10240,
            cacheGroups: {
                ...vendorCacheGroups,
                webphone: {
                    test: /([\\/]components[\\/]Webphone[\\/])|([\\/]jssip[\\/])/,
                    name: 'webphone'
                }
            },
        }
    },
    stats: {
        assetsSort: '!size',
        colors: true,
        logging: 'error',
        reasons: false,
        children: false,
        usedExports: false
    }
});
