import * as React from 'react';

import { ONE_SECOND } from '../../constants';
import { ISessionInfo, SessionHistoryInfoHandler } from '../../models/session';
import FormatDate, { FormatDateInString } from '../../ui/FormatDate';
import { Link } from '../../ui/Link';
import { isObjectEqual } from '../../utils/isObjectEqual';
import Img from '../Img';
import { PictureGallery } from '../PictureGallery';
import * as style from './index.css';

interface IAcceptancePhoto {
    link: string;
    preview_url?: string;
    session_id: string;
    timestamp: number;
    user_id: string;
}

export interface IAcceptancePhotos {
    groups: {
        links: IAcceptancePhoto[];
        tags: string[];
        timestamp: number;
    }[];
    links: IAcceptancePhoto[];
}

interface IAcceptancePhotosProps {
    data?: IAcceptancePhotos;
    showSessionLink?: boolean;
    session?: ISessionInfo;
}

interface IAcceptancePhotosState {
    showImgIndex: number | null;
    photos: { link: string; title: string }[];
    content: { title: string; groupContent: JSX.Element[] }[];
    contentLength: number;
}

const lazy_image_classname = 'lzy_img';
const lazy_image_root = 'root_acceptance_photos';

export default class AcceptancePhotos extends React.Component<IAcceptancePhotosProps, IAcceptancePhotosState> {
    state: IAcceptancePhotosState = {
        showImgIndex: null,
        photos: [],
        content: [],
        contentLength: 0,
    };

    imgIndex = 0;

    openImg(showImgIndex: number) {
        this.setState({
            showImgIndex,
        });
    }

    onClose() {
        this.setState({
            showImgIndex: null,
        });
    }

    shouldComponentUpdate(
        nextProps: Readonly<IAcceptancePhotosProps>,
        nextState: Readonly<IAcceptancePhotosState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    isAfterSession(imageTimeStamp: number, sessionFinish: number): boolean {
        return imageTimeStamp > sessionFinish;
    }

    extractData(links, photos, content, title) {
        const { showSessionLink } = this.props;
        const groupContent: JSX.Element[] = [];

        const finish = this.props.session && SessionHistoryInfoHandler.getFinish.call(this.props.session);

        links?.reverse().forEach((img, index) => {
            const afterSession = finish && this.isAfterSession(img.timestamp * ONE_SECOND, finish);

            photos.push({
                link: img.link,
                title: FormatDateInString({ value: img.timestamp * ONE_SECOND }),
            });

            groupContent.push(
                <AcceptancePhoto className={`${lazy_image_classname}`
                    + ` ${afterSession ? style.after_session : ''}`}
                                 showSessionLink={showSessionLink}
                                 img={img}
                                 key={`tag_${index}`}
                                 index={this.imgIndex + index}
                                 openImg={this.openImg.bind(this)}>
                    {showSessionLink &&
                    <Link href={`#/session/${img.session_id}?back=${encodeURIComponent(location.hash)}`}>Сессия</Link>}
                    {afterSession ? <span title={'Фото после сесии'}>🏁</span> : ''}
                </AcceptancePhoto>,
            );
        });

        this.imgIndex += links.length;

        content.push({
            title,
            groupContent,
        });
    }

    prepareData() {
        const { data } = this.props;

        const photos: { link: string; title: string }[] = [];
        const contentLength = (data && data.links && data.links.length || 0)
            + (data && data.groups && data.groups.length || 0);

        const content: { title: string; groupContent: JSX.Element[] }[] = [];

        data && data.links && data.links.length && this.extractData(data.links || [], photos, content, '');

        data && data.groups && data.groups.length && data.groups.forEach((group) => {
            this.extractData(group.links || [], photos, content, `Фидбек: ${group.tags.join(', ')}`);
        });
        this.setState({
            photos,
            content,
            contentLength,
        });
    }

    componentDidMount() {
        this.prepareData();
    }

    render() {
        const {
            photos,
            content,
            contentLength,
        } = this.state;

        return <>
            {contentLength
                ? <div className={lazy_image_root}>
                    {content.map((group, index) => (
                        <React.Fragment key={`group_${index}`}>
                            <h3 className={style.group_title}>{group.title}</h3>
                            <div className={`${style.acceptance_photos}`}>
                                {group.groupContent}
                            </div>
                        </React.Fragment>
                    ))}
                </div>
                : <h4>Фото не найдено</h4>
            }
            {this.state.showImgIndex !== null
                ? <PictureGallery initialIndex={this.state.showImgIndex || 0}
                                  pictures={photos}
                                  onClose={this.onClose.bind(this)}/>
                : null
            }
        </>;
    }
}

interface IAcceptancePhotoProps {
    img: IAcceptancePhoto;
    index: number;
    openImg: () => void;
    showSessionLink?: boolean;
    className: string;
}

export const AcceptancePhoto = React.memo((props: React.PropsWithChildren<IAcceptancePhotoProps>) => {
    const { img, index, openImg, children, className } = props;

    return <div className={`${style.wrap_img_acceptance_photos} ${className || ''}`} key={index}>
        <div className={style.image_container}>
            <Link onClick={openImg.bind(null, index)}>
                <Img src={img.preview_url || img.link}/>
            </Link>
        </div>
        <div className={style.photo_info}>
            <div className={style.formatted_date}>
                <FormatDate value={img.timestamp * ONE_SECOND} withSecond/>
            </div>
            <div>
                {children}
            </div>
        </div>
    </div>;
});
