import * as React from 'react';

import { Button, CancelButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import coreStyles from '../../../../ui/index.css';
import { QuickActionButton } from '../../../../ui/QuickActionButton';
import TextArea from '../../../../ui/TextArea';
import { Request2 } from '../../../../utils/request';
import { REQUESTS } from '../request';
import styles from './index.css';

interface IGarageXLSXProps {
    request: Request2;
    refreshData: () => void;
}

export const GarageXLSX = (props: IGarageXLSXProps) => {
    const { request, refreshData } = props;
    const [latestMutationId, setLatestMutationId] = React.useState<string | null>(null);
    const [isModalOpen, openModal] = React.useState<boolean>(false);
    const [fileName, setFileName] = React.useState<string | null>(null);
    const [comment, setComment] = React.useState<string>('');
    const [file, setFile] = React.useState<Blob | null>(null);
    const [error, setError] = React.useState<Error | null>(null);
    const [isLoading, setLoading] = React.useState<boolean>(false);

    const selectXLSX = (data: FileList) => {
        setFileName(data[0].name);
        setFile(data[0]);
    };

    const sendXLSX = async() => {
        setLoading(true);
        const formData = new FormData();
        if (file) {
            formData.append('file', file);
            formData.append('comment', comment);
            try {
                const { mutation_id } = await request.exec(REQUESTS.UPLOAD_GARAGE_XLSX, {
                    formData,
                });

                setLatestMutationId(mutation_id);
                setLoading(false);
                openModal(false);
                refreshData();
            } catch(error) {
                setLoading(false);
                setError(error);
            }
        }
    };

    const rollbackMutation = async () => {
        return await request.exec(REQUESTS.ROLLBACK_MUTATION_ID, {
            queryParams: {
                ID: latestMutationId,
            },
        }).then(() => {
            setLatestMutationId(null);
            refreshData();
        });
    };

    return <>
        {isModalOpen
            ? <Window error={error}
                      title={'Загрузить xlsx'}
                      onClose={() => openModal(false)}>
                <div className={styles.fileUpload}>
                    <Button file
                            fileAccept={'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'}
                            onClick={selectXLSX}>
                        Выбрать файл
                    </Button>
                    <span>
                        {fileName
                            ? fileName
                            : 'Файл не выбран'
                        }
                    </span>
                </div>
                <TextArea value={comment} onChange={setComment} placeholder={'Комментарий'}/>
                <div className={coreStyles.button_container}>
                    <CancelButton onClick={() => openModal(false)}/>
                    <Button isLoading={isLoading}
                            disabled={!file}
                            onClick={sendXLSX}>
                        Отправить
                    </Button>
                </div>
            </Window>
            : null
        }
        <div className={styles.buttons}>
            <Button onClick={() => openModal(true)}>
                Загрузить XLSX
            </Button>
            {latestMutationId
                ? <QuickActionButton className={styles.rollbackButton}
                                     question={`Откатить изменениe ${latestMutationId} из XLSX?`}
                                     showConfirm
                                     button={{
                                         children: 'Откатить изменения из XLSX',
                                     }}
                                     accept={rollbackMutation}/>
                : null
            }
        </div>
    </>;
};
