import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import { useRequestHandler } from '../../../hooks/useRequestHandler';
import { Button } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import { CenteredSpin } from '../../Spin';
import AnalyticsGarageItem from './AnalyticsGarageItem';
import GarageXLSX from './GarageXLSX';
import styles from './index.css';
import { ANALYICS_GARAGE_REQUESTS, REQUESTS } from './request';
import { IVinResponse } from './types';

const LIMIT = 100;

export const AnalyticsGarage = React.memo(() => {
    const [vin, setVin] = React.useState<string | null>(null);
    const [timestamp, setTimestamp] = React.useState<number | null>(null);
    const [searchLimit, setSearchLimit] = React.useState<number>(LIMIT);
    const [limit, setLimit] = React.useState<number>(LIMIT);
    const [searchVin, setSearchVin] = React.useState<string>('');
    const [searchTimestamp, setSearchTimestamp] = React.useState<number | null>(null);
    const [shouldRefreshData, refreshData] = React.useState<boolean>(true);
    const request = React.useMemo(() => new Request2({ requestConfigs: ANALYICS_GARAGE_REQUESTS }), []);

    const vinInfoOptions = React.useMemo(() => {
        refreshData(false);

        return {
            requestName: REQUESTS.GET_VIN_INFO,
            requestOptions: {
                queryParams: {
                    begin: vin,
                    time: timestamp ? Math.floor(timestamp / ONE_SECOND) : null,
                    limit: limit,
                },
            },
        };
    }, [vin, timestamp, limit, shouldRefreshData]);

    const search = () => {
        if (searchVin) {
            setVin(searchVin);
        } else {
            setVin(null);
        }

        if (searchTimestamp) {
            setTimestamp(searchTimestamp);
        } else {
            setTimestamp(null);
        }

        if (searchVin) {
            setLimit(1);
        } else if (searchLimit) {
            setLimit(searchLimit);
        } else {
            setLimit(LIMIT);
            setSearchLimit(LIMIT);
        }
    };

    const [
        isLoading,
        vinResponse,
        error,
        getGarageAnalytics,
    ] = useRequestHandler<IVinResponse>(request, vinInfoOptions);

    React.useEffect(() => {
        getGarageAnalytics();
    }, []);

    return <>
        <div className={styles.garageHeader}>
            <div className={styles.garageSearch}>
                <Input value={searchVin}
                       placeholder={'VIN'}
                       onChange={setSearchVin}/>
                <DatePicker value={searchTimestamp}
                            placeholder={'Время снепшота'}
                            onChange={setSearchTimestamp}/>
                <Input value={searchLimit}
                       placeholder={'Записей на странице'}
                       onChange={setSearchLimit}/>
                <Button onClick={search}>
                    Поиск
                </Button>
            </div>
            <GarageXLSX refreshData={() => refreshData(true)}
                        request={request}/>
        </div>
        {error
            ? <SimpleError error={error}/>
            : isLoading
                ? <CenteredSpin/>
                : <>
                    {vinResponse?.infos?.length
                        ? <>
                            <div className={styles.garageTableContainer}>
                                <table>
                                    <thead>
                                        <tr className={`${styles.garageItem} ${styles.garageTableHeader}`}>
                                            <td>VIN</td>
                                            {vinResponse?.fields
                                                ?.map(({ title }, index) => {
                                                    return <td key={index}>{title}</td>;
                                                })}
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {
                                            vinResponse?.infos.map((item, index) => {
                                                return <AnalyticsGarageItem key={index}
                                                                            item={item}
                                                                            fields={vinResponse?.fields}/>;
                                            })
                                        }
                                    </tbody>
                                </table>
                            </div>
                            {vinResponse?.next_begin?.length &&
                                <Link className={styles.nextPage}
                                      onClick={() => setVin(vinResponse!.next_begin)}>
                                    Следующая страница
                                </Link>
                            }
                        </>
                        : <NoInformation/>
                    }
                </>
        }
    </>;
});
