import React from 'react';

import { ONE_SECOND } from '../../../constants';
import { Link } from '../../../ui/Link';
import { Request2 } from '../../../utils/request';
import { Translate } from '../../../utils/translate';
import { IStore } from '../../App/store';
import { getTranslateKey, PROGRESS_STATUSES } from '../Tree/constants';
import { Navigation } from '../Tree/Navigation/component';
import { ObjectView } from '../Tree/ObjectView/component';
import { TaskHistory } from '../Tree/TaskHistory/component';
import { OBJECTS_ID_PARAM } from '../Tree/types';
import style from './index.css';
import { CORE_DRIVE_LENS_REQUESTS as requestConfigs, REQUESTS } from './request';

interface ITask {
    ID: number;
    ActionID: number;
    OwnerID: string;
    Status: string;
    CreateTime: number;
    UpdateTime: number;
    Settings: { [key: string]: IOption };
    Options: { [key: string]: IOption };
    PyData: { [key: string]: any; resources: any[] };
}

interface IOption {
    Type: string;
    Title: string;
    Description: string;
    Settings: { [key: string]: any };
    Value: { [key: string]: any } | number | string;
    Visible: boolean;
    Editable: boolean;
}

type IDriveLensProps = IStore

interface IDriveLensState {
    dirID: number;
    currentObject: {
        type: string | null;
        id: string;
    };
    tasks: ITask[];
    endPage: number;
    currentPage: number;
    perPage: number;
    error: Error | null;
}

const DENIED_ROOT_ID = 1;
const DELAY_IN_SEC = 5;
const DELAY = ONE_SECOND * DELAY_IN_SEC;

export default class DriveLens extends React.Component<IDriveLensProps, IDriveLensState> {
    request = new Request2({ requestConfigs });
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;
    timer: any;
    state = {
        currentObject: {
            type: null,
            id: '',
        },
        dirID: 4,
        tasks: [],
        currentPage: 0,
        endPage: 0,
        perPage: 0,
        error: null,
    };

    componentDidMount(): void {
        this.getTasksList();
    }

    getTasksList(pagination = 0): void {
        clearTimeout(this.timer);
        const page = this.state.currentPage + pagination;
        this.request.exec(REQUESTS.GET_RUNS_LIST, { queryParams: { Page: page } })
            .then(response => {
                const { Items: tasks = [], EndPage: endPage, PerPage: perPage, Page: currentPage } = response;
                const isProgress = tasks.some(task => PROGRESS_STATUSES.includes(task.Status));
                isProgress && (this.timer = setTimeout(this.getTasksList.bind(this, 0), DELAY));
                this.setState({ tasks, endPage, perPage, currentPage });
            })
            .catch((error) => {
                this.setState({ error });
            });
    }

    componentWillUnmount() {
        clearInterval(this.timer);
    }

    handleObjectClick(type: string, id) {
        switch (type) {
        case OBJECTS_ID_PARAM.dir: {
            id !== DENIED_ROOT_ID && this.setState({ dirID: id });
            break;
        }

        case OBJECTS_ID_PARAM.action: {
            this.setState({ currentObject: { type: OBJECTS_ID_PARAM.action, id } });
            break;
        }

        case OBJECTS_ID_PARAM.task:
            this.setState({ currentObject: { type: OBJECTS_ID_PARAM.task, id } });
            break;
        case OBJECTS_ID_PARAM.runAction: {
            this.setState({ currentObject: { type: OBJECTS_ID_PARAM.runAction, id } });
            break;
        }
        }
    }

    render() {
        const { perPage, currentPage, endPage, tasks, dirID, currentObject } = this.state;
        const previousIndex = perPage * currentPage;

        return <div className={style['drive-lens__container']}>
            <Navigation className={style['drive-lens__navigation']}
                        dirID={dirID}
                        deniedRootID={DENIED_ROOT_ID}
                        handleObjectClick={this.handleObjectClick.bind(this)}/>
            <div className={style['drive-lens']}>
                {currentObject.type &&
                <ObjectView t={this.t}
                            getTasksList={this.getTasksList.bind(this)}
                            handleObjectClick={this.handleObjectClick.bind(this)}
                            taskID={currentObject.type === OBJECTS_ID_PARAM.task && currentObject.id || null}
                            actionID={currentObject.type === OBJECTS_ID_PARAM.action && currentObject.id || null}
                            runActionID={currentObject.type === OBJECTS_ID_PARAM.runAction
                                && currentObject.id
                                || null}/>}
                <TaskHistory t={this.t}
                             handleObjectClick={this.handleObjectClick.bind(this)}
                             tasks={tasks}
                             previousIndex={previousIndex}/>
                {endPage > 1 &&
                <div className={style.pagination}>
                    <Link className={currentPage === 0 && style.pagination__link_hidden}
                          onClick={this.getTasksList.bind(this, -1)}>
                        {this.t.getItem(getTranslateKey('prev'))}
                    </Link>
                    <div className={style.pagination__number}>
                        {currentPage + 1}
                    </div>
                    <Link className={currentPage >= endPage - 1 && style.pagination__link_hidden}
                          onClick={this.getTasksList.bind(this, 1)}>
                        {this.t.getItem(getTranslateKey('next'))}
                    </Link>
                </div>}
            </div>
        </div>;
    }
}
