/*eslint-disable*/
import React from 'react';
import * as THREE from 'three';
import DatePicker from '../../../ui/DatePicker';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Request2 } from '../../../utils/request';
import { CORE_HEAT_MAP_REQUESTS, REQUESTS } from './request';
import Select from '../../../ui/Select';
import Spin from '../../Spin';
import { debounce, pad } from '../../../utils/utils';
import { initMap } from '../../MainMap/utils';
import { ONE_DAY, ONE_HOUR, ONE_MINUTE, ONE_SECOND } from '../../../constants';
import { SimpleError } from '../../SimpleError';
import style from './index.css';
import PlayIcon from '../../../../svg-components/play.component.svg';
import PauseIcon from '../../../../svg-components/pause.component.svg';

declare const ymaps: any;

interface IFrame {
    Time: number,
    Tiles: [number, number][],
    Arrows: [number, number, number][]
}

interface IHeatMapState {
    error: Error | null,
    start: number[],
    tileSize: number[],
    stepSize: number[],
    frames: IFrame[],
    isPlay: boolean,
    date: {
        from: number,
        until: number
    },
    defaultLayer: string,
    layers: {
        [key: string]: {
            Title: string
        }
    },
    isLoading: boolean
}

class HeatMap extends React.Component<{}, IHeatMapState> {

    state: IHeatMapState = {
        error: null,
        frames: [],
        start: [],
        tileSize: [],
        stepSize: [],
        isPlay: false,
        date: {
            from: Date.now() - ONE_DAY * 3,
            until: Date.now() - ONE_DAY * 2
        },
        layers: {},
        defaultLayer: '',
        isLoading: false,
    };

    request = new Request2({requestConfigs: CORE_HEAT_MAP_REQUESTS});
    inputRange: React.RefObject<HTMLInputElement> = React.createRef();
    currentTime: React.RefObject<HTMLDivElement> = React.createRef();

    time = 0;
    interval;

    map;
    projection;
    zoom = 10;
    width;
    height;
    globalPixelCenter;
    mapGlobalPixelCenter;
    lastGlobalPixelCenter;

    scene;
    camera;
    renderer;

    componentDidMount() {
        initMap('map', (map) => {
            this.map = map;
            const mapSize = this.map.container.getSize();
            this.projection = this.map.options.get('projection');
            this.globalPixelCenter = this.map.getGlobalPixelCenter();
            this.mapGlobalPixelCenter = this.globalPixelCenter;
            this.lastGlobalPixelCenter = this.globalPixelCenter;
            const bound = this.map.getBounds();
            this.width = mapSize[0];
            this.height = mapSize[1];
            this.getLayers(bound);
            this.initCanvas();
            this.initListener();
            this.checkOffsetPosition();
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
        this.map && this.map.destroy();
    }

    getLayers(bound) {
        this.request.exec(REQUESTS.GET_LAYERS)
            .then(res => {
                this.getFrames(bound, res.DefaultLayer);
                this.setState({layers: res.Layers, defaultLayer: res.DefaultLayer});
            })
            .catch(error => {
                this.setState({error});
            });
    }

    getFrames(bound, layer) {
        this.map.behaviors.disable('scrollZoom');
        !this.state.isLoading && this.setState({isLoading: true, error: null}, () => {
            this.request.exec(REQUESTS.GET_FRAMES, {
                headers: {'Content-Type': 'application/json'},
                body: {
                    Layer: layer,
                    Dates: this.getDatesFromRange(this.state.date.from, this.state.date.until),
                    Box: [bound[0], [bound[1][0] - bound[0][0], bound[1][1] - bound[0][1]]]
                }
            })
                .then(data => {
                        this.setState({
                            frames: data.Frames,
                            start: data.Start,
                            tileSize: data.TileSize,
                            stepSize: data.StepSize,
                            isLoading: false
                        }, () => {
                            this.draw(this.time);
                            this.map.behaviors.enable('scrollZoom');
                        });
                    }
                )
                .catch(error => {
                    this.setState({error, isLoading: false});
                });
        });
    }

    getDatesFromRange(from, until) {
        let result: string[] = [];
        while (from <= until) {
            result.push(new Date(from).toISOString().slice(0, 10));
            from += ONE_DAY;
        }
        return result;
    }

    initCanvas() {
        this.scene = new THREE.Scene();
        this.camera = new THREE.OrthographicCamera(
            this.width / -2, this.width / 2, this.height / 2, this.height / -2, 1, 1000
        );
        this.renderer = new THREE.WebGLRenderer({antialias: true, alpha: true});
        this.renderer.setPixelRatio(window.devicePixelRatio);
        this.renderer.setSize(this.width, this.height);
        const pane = new ymaps.pane.StaticPane(this.map, {zIndex: 100});
        pane.getElement().appendChild(this.renderer.domElement);
        this.map.panes.append('canvas', pane);
        this.camera.position.z = 200;
    }

    initListener() {
        this.map.events.add('actiontick', this.handleTick.bind(this));
        this.inputRange.current && this.inputRange.current.addEventListener('input', this.handleInput.bind(this));
        this.map.events.add('sizechange', this.handleResize.bind(this));
        this.checkOffsetPosition = debounce(this.checkOffsetPosition, ONE_SECOND * 2);
        this.reloadingFrames = debounce(this.reloadingFrames, ONE_SECOND);
    }

    handleResize(e) {
        [this.width, this.height] = e.get('newSize');
        this.renderer.setSize(this.width, this.height);
        this.camera.left = this.width / -2;
        this.camera.right = this.width / 2;
        this.camera.top = this.height / 2;
        this.camera.bottom = this.height / -2;
        this.camera.updateProjectionMatrix();
        this.state.frames && this.draw(this.time);
    }

    handleInput(e) {
        const time = +e.target.value;
        this.time = time;
        this.updateTime();
        this.draw(time);
    }

    handleInputDate(type, value) {
        this.setState({date: {...this.state.date, [type]: value}}, () => {
            this.getFrames(this.map.getBounds(), this.state.defaultLayer);
        });
    }

    handleSelectLayer(value) {
        this.setState({defaultLayer: value}, () => {
            this.getFrames(this.map.getBounds(), value);
        });
    }

    updateTime() {
        if (this.inputRange.current) {
            this.inputRange.current.value = this.time.toString();
        }
        if (this.currentTime.current) {
            const currentTime = this.state.frames && this.state.frames[this.time].Time || 0;
            this.currentTime.current.innerText =
                `${pad(Math.trunc(currentTime / (ONE_HOUR / ONE_SECOND)), 2)}:` +
                `${pad((currentTime % (ONE_HOUR / ONE_SECOND)) / (ONE_MINUTE / ONE_SECOND), 2)}`;
        }
    }

    togglePlay() {
        this.state.frames && this.state.frames.length && this.setState({isPlay: !this.state.isPlay}, () => {
            if (this.state.isPlay) {
                this.interval = setInterval(() => {
                    if (this.time + 1 > this.state.frames.length - 1) {
                        this.setState({isPlay: false});
                        clearInterval(this.interval);
                        this.time = 0;
                        this.updateTime();
                    } else {
                        this.time++;
                        this.updateTime();
                        this.draw(this.time);
                    }
                }, ONE_SECOND);
            } else {
                clearInterval(this.interval);
            }
        });
    }

    reloadingFrames() {
        this.getFrames(this.map.getBounds(), this.state.defaultLayer);
    }

    handleTick(e) {
        let {zoom, globalPixelCenter} = e.get('tick');
        this.globalPixelCenter = globalPixelCenter;
        if (this.zoom !== zoom) {
            this.zoom = zoom;
            this.mapGlobalPixelCenter = globalPixelCenter;
            this.lastGlobalPixelCenter = globalPixelCenter;
            this.draw(this.time);
            this.reloadingFrames();
        }
        this.checkOffsetPosition();
        const x = this.mapGlobalPixelCenter[0] - globalPixelCenter[0];
        const y = this.mapGlobalPixelCenter[1] - globalPixelCenter[1];
        this.moveCamera(x, y);
    }

    checkOffsetPosition() {
        if (
            Math.abs(this.lastGlobalPixelCenter[0] - this.globalPixelCenter[0]) > this.width / 4 ||
            Math.abs(this.lastGlobalPixelCenter[1] - this.globalPixelCenter[1]) > this.height / 4
        ) {
            this.lastGlobalPixelCenter = this.globalPixelCenter;
            this.getFrames(this.map.getBounds(), this.state.defaultLayer);
        }
    }

    draw(time) {
        this.clearThree(this.scene);
        const {frames, start, tileSize, stepSize} = this.state;
        frames && frames[time].Tiles && frames[time].Tiles.forEach(tile => {
            const squareX = tile[0] % (1 << 16);
            const squareY = ~~(tile[0] / (1 << 16));

            const coord1 = (this.projection.toGlobalPixels([start[0] + squareX * tileSize[0], start[1] + squareY * tileSize[1]], this.zoom));
            const coord2 = (this.projection.toGlobalPixels([start[0] + (squareX + 1) * tileSize[0], start[1] + (squareY + 1) * tileSize[1]], this.zoom));

            const squareWidth = coord2[0] - coord1[0];
            const squareHeight = coord1[1] - coord2[1];
            const squareGeometry = new THREE.PlaneBufferGeometry(squareWidth, squareHeight);
            const squareMaterial = new THREE.MeshBasicMaterial(
                {
                    color: 0x303F9F,
                    opacity: (1 + tile[1]) / 256,
                    transparent: true
                }
            );
            const square = new THREE.Mesh(squareGeometry, squareMaterial);
            const x = coord1[0] - this.mapGlobalPixelCenter[0] + squareWidth / 2;
            const y = coord1[1] - this.mapGlobalPixelCenter[1] + squareHeight / 2;
            square.position.x = x;
            square.position.y = -y;
            this.scene.add(square);
        });


        frames && frames[time].Arrows && frames[time].Arrows.forEach(arrow => {
            const beginLat = start[0] + (arrow[0] % (1 << 16)) * stepSize[0];
            const beginLon = start[1] + (~~(arrow[0] / (1 << 16))) * stepSize[1];

            const endLat = start[0] + (arrow[1] % (1 << 16)) * stepSize[0];
            const endLon = start[1] + (~~(arrow[1] / (1 << 16))) * stepSize[1];

            const begin = this.projection.toGlobalPixels([beginLat, beginLon], this.zoom);
            const end = this.projection.toGlobalPixels([endLat, endLon], this.zoom);

            const x1 = begin[0] - this.mapGlobalPixelCenter[0];
            const y1 = this.mapGlobalPixelCenter[1] - begin[1];

            const x2 = end[0] - this.mapGlobalPixelCenter[0];
            const y2 = this.mapGlobalPixelCenter[1] - end[1];

            const height = Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));

            const x3 = (x1 + x2) / 2;
            const y3 = (y1 + y2) / 2;
            const value = this.zoom < 10 ? (arrow[2] + 20) / (20 - this.zoom) : (arrow[2] + 20) / 10;

            const geometry = new THREE.PlaneBufferGeometry(value, height);
            const material = new THREE.MeshBasicMaterial({color: 0xff5722, transparent: true, opacity: 0.5});
            const plane = new THREE.Mesh(geometry, material);
            plane.position.x = x3;
            plane.position.y = y3;
            plane.position.z = 1;
            plane.rotation.z = Math.atan2(y2 - y1, x2 - x1) + Math.PI / 2;
            this.scene.add(plane);

            const triangleGeometry = new THREE.Geometry();
            triangleGeometry.vertices.push(new THREE.Vector3(0, height / 2 + value * 1.5, 0));
            triangleGeometry.vertices.push(new THREE.Vector3(-value * 0.8, height / 2, 0));
            triangleGeometry.vertices.push(new THREE.Vector3(value * 0.8, height / 2, 0));
            triangleGeometry.faces.push(new THREE.Face3(0, 1, 2));
            const triangleMaterial = new THREE.MeshBasicMaterial({
                color: 0xff5722,
                opacity: 0.5,
                transparent: true
            });
            const triangle = new THREE.Mesh(triangleGeometry, triangleMaterial);
            triangle.position.x = x3;
            triangle.position.y = y3;
            triangle.position.z = 1;
            triangle.rotation.z = Math.atan2(y2 - y1, x2 - x1) - Math.PI / 2;
            this.scene.add(triangle);
        });
        this.renderer.render(this.scene, this.camera);
    }

    moveCamera(x, y) {
        this.camera.position.x = -x;
        this.camera.position.y = y;
        this.renderer.render(this.scene, this.camera);
    }

    clearThree(obj) {
        while (obj.children.length > 0) {
            this.clearThree(obj.children[0]);
            obj.remove(obj.children[0]);
        }
        if (obj.geometry) {
            obj.geometry.dispose();
        }
        if (obj.material) {
            obj.material.dispose();
        }
        if (obj.texture) {
            obj.texture.dispose();
        }
    }

    render() {
        return <div className={style.map}>
            <div id="map" className={style.map__ymap}/>
            {this.state.isLoading && <Spin className={style.map__spin}/>}
            {this.state.error &&
            <div className={style['map__error-container']}>
                <div className={style.map__error}>
                    <SimpleError error={this.state.error}/>
                </div>
            </div>}
            <div className={`${style.map__controls} ${style['map__controls_top-left']}`}>
                <DatePicker
                    onlyDate={true}
                    placeholder={'Начало периода'}
                    disabled={this.state.isLoading}
                    value={this.state.date.from}
                    onChange={this.handleInputDate.bind(this, 'from')}
                />
                <DatePicker
                    onlyDate={true}
                    placeholder={'Конец периода'}
                    disabled={this.state.isLoading}
                    value={this.state.date.until}
                    onChange={this.handleInputDate.bind(this, 'until')}
                />
                <div className={style.map__select}>
                    <Select
                        placeholder={'Слой'}
                        initialValues={[this.state.defaultLayer]}
                        disabled={this.state.isLoading}
                        options={Object.entries(this.state.layers).map(item => ({
                            value: item[0],
                            text: item[1].Title
                        }))}
                        onSelect={this.handleSelectLayer.bind(this)}
                    />
                </div>
            </div>
            <div className={`${style.map__controls} ${style['map__controls_top-right']}`}>
                <div className={style.map__time} ref={this.currentTime}>00:00</div>
            </div>
            <div className={`${style.map__controls} ${style.map__controls_bottom}`}>
                <Button
                    className={style.map__play}
                    onClick={this.togglePlay.bind(this)}
                    colorType={ButtonTypes.positive}>
                    {this.state.isPlay
                        ? <PauseIcon />
                        : <PlayIcon />}
                </Button>
                <input
                    className={style.map__input}
                    min={0}
                    max={this.state.frames && this.state.frames.length - 1 || 0}
                    type="range"
                    ref={this.inputRange}
                />
            </div>
        </div>;
    }
}

export default HeatMap;
