import React, { useEffect, useState } from 'react';

import { ONE_DAY, ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import DatePicker from '../../../ui/DatePicker';
import { toUTCDateTime } from '../../../ui/FormatDate';
import { Confirm } from '../../../ui/FullModal';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import { FLOAT } from '../Tree/LogicalJSONEditor/SimpleJSONEQueryOperand/component';
import { Table } from '../ui/Table';
import styles from './index.css';
import { REGISTRY_REQUESTS, REQUESTS } from './request';

interface IColumn {
    Type: string;
    Required: boolean;
}

interface IRow {
    [key: string]: any;
}

interface ISnapshot {
    Table: string;
    Columns: {
        [key: string]: IColumn;
    };
    Rows: IRow[];
    Time: number;
}

const TABLES = [
    { id: 'fueling', name: 'Заправки', skeletonColumn: 2 },
    { id: 'total_costs', name: 'Прочие расходы', skeletonColumn: 2 },
    { id: 'parking', name: 'Парковки', skeletonColumn: 2 },
    { id: 'aeroports', name: 'Аэропорты', skeletonColumn: 2 },
    { id: 'fuel_types', name: 'Тип топлива по моделям', skeletonColumn: 2 },
    { id: 'cleaning', name: 'Мойки', skeletonColumn: 6 },
];

let preparedData = {};
let preparedInfo = {};

const Registry = () => {
    const request = new Request2({ requestConfigs: REGISTRY_REQUESTS });

    const [time, setTime] = useState(Math.round(~~(Date.now() / ONE_DAY) * ONE_DAY));
    const [state, setState] = useState({} as { [key: string]: ISnapshot });
    const [loadInfo, setLoadInfo] = useState({});
    const [confirmInfo, setConfirmInfo] = useState({
        table: '',
        isOpen: false,
        isDeleting: false,
        error: null,
    });

    useEffect(() => {
        TABLES.forEach(table => getTable(table.id, Math.round(time / ONE_SECOND)));

        return () => {
            request.abort();
        };
    }, []);

    const getTable = (table, time) => {
        preparedInfo = {
            ...preparedInfo,
            [table]: {
                ...preparedInfo[table],
                isLoading: true,
            },
        };
        setLoadInfo(preparedInfo);
        request.exec(REQUESTS.GET_REGISTRY, {
            body: {
                Table: table,
                FromTime: time,
            },
        }).then(response => {
            preparedData = { ...preparedData, [table]: response };
            preparedInfo = {
                ...preparedInfo,
                [table]: {
                    ...preparedInfo[table],
                    isLoading: false,
                    loadingError: null,
                },
            };
            setState(preparedData);
            setLoadInfo(preparedInfo);
        }).catch(error => {
            preparedInfo = {
                ...preparedInfo,
                [table]: {
                    ...preparedInfo[table],
                    isLoading: false,
                    loadingError: error,
                },
            };
            setLoadInfo(preparedInfo);
        });
    };

    const updateTable = (rows, table) => {
        const newState = { ...state };
        Object.keys(state[table]?.Columns).forEach(key => {
            if (state[table].Columns[key].Type === FLOAT) {
                rows.forEach(row => row[key] = +row[key]);
            }
        });

        newState[table].Rows = rows;
        setState(newState);
        setLoadInfo({
            ...loadInfo,
            [table]: {
                ...loadInfo[table],
                isUpdating: true,
            },
        });

        request.exec(REQUESTS.UPDATE_REGISTRY, {
            body: {
                Table: table,
                Columns: state[table].Columns,
                Rows: rows,
                Time: Math.round(time / ONE_SECOND),
            },
        }).then(() => {
            setLoadInfo({
                ...loadInfo,
                [table]: {
                    ...loadInfo[table],
                    isUpdating: false,
                    updateError: null,
                },
            });
            getTable(table, Math.round(time / ONE_SECOND));
        }).catch(error => {
            setLoadInfo({
                ...loadInfo,
                [table]: {
                    ...loadInfo[table],
                    updateError: error,
                },
            });
        });
    };

    const removeTable = (table) => {
        setConfirmInfo({ ...confirmInfo, isDeleting: true });
        request.exec(REQUESTS.DELETE_REGISTRY, {
            body: {
                Table: table,
                Time: state[table].Time,
            },
        }).then(() => {
            setConfirmInfo({ ...confirmInfo, isOpen: false });
            getTable(table, time);
        }).catch(error => {
            setConfirmInfo({ ...confirmInfo, isDeleting: false, error });
        });
    };

    const onDatePickerChange = (dateValue) => {
        setTime(dateValue);
        TABLES.forEach(table => getTable(table.id, dateValue / ONE_SECOND));
    };

    return <>
        <DatePicker utc
                    value={time}
                    className={styles.registry__datePicker}
                    onChange={onDatePickerChange}
                    placeholder={'Введите время UTC'}/>
        {TABLES.map(table => <div key={table.id}>
            <div className={styles.registry__table}>
                <div className={styles.table__title}>{table.name} {state[table.id]?.Time
                    && state[table.id].Time != 0
                    && <>(цены от {toUTCDateTime(state[table.id].Time * ONE_SECOND)})
                        <Button basic
                                className={styles.registry__button_remove}
                                colorType={ButtonTypes.negative}
                                onClick={() => {
                                    setConfirmInfo({ ...confirmInfo, table: table.id, isOpen: true });
                                }}>Удалить цены</Button>
                    </> || ''}

                </div>
                <div className={styles.table__wrapper}>
                    {loadInfo[table.id]?.loadingError
                        ? <SimpleError error={loadInfo[table.id]?.loadingError}/>
                        : <>
                            <Table head={Object.keys(state[table.id]?.Columns || [])}
                                   body={state[table.id]?.Rows || []}
                                   isLoading={loadInfo[table.id]?.isLoading}
                                   isUpdating={loadInfo[table.id]?.isUpdating}
                                   onUpdate={(rows) => updateTable(rows, table.id)}
                                   skeletonColumn={table.skeletonColumn}/>
                            {loadInfo[table.id]?.updateError
                                && <SimpleError error={loadInfo[table.id].updateError}/>}
                        </>
                    }
                </div>
            </div>
        </div>,
        )}
        {confirmInfo.isOpen
            ? <Confirm question={'Вы действительно хотите удалить?'}
                       error={confirmInfo.error}
                       accept={removeTable.bind(null, confirmInfo.table)}
                       isWorking={confirmInfo.isDeleting}
                       onClose={setConfirmInfo.bind(null, { ...confirmInfo, isOpen: false })}/>
            : null}
    </>;
};

export default Registry;
