import React from 'react';
import ReactTooltip from 'react-tooltip';

import { ONE_SECOND } from '../../../../../constants';
import Checkbox from '../../../../../ui/Checkbox';
import DatePicker from '../../../../../ui/DatePicker';
import { Input } from '../../../../../ui/Input';
import Select from '../../../../../ui/Select';
import { IOption } from '../../types';
import style from './index.css';

const STRING = 'String';
const INTEGER = 'Integer';
export const FLOAT = 'Float';
const ENUM = 'Enum';
const ARRAY = 'Array';
const TIMESTAMP = 'Timestamp';
const BOOLEAN = 'Boolean';
const FIELD = 'field';

export const NIL_OPERATORS = {
    IsNil: { value: 'IsNil', text: 'Не задано' },
    IsNotNil: { value: 'IsNotNil', text: 'Задано' },
};

const OPERATORS = {
    Equal: { value: 'Equal', text: '=' },
    Less: { value: 'Less', text: '<' },
    Greater: { value: 'Greater', text: '>' },
    LessEqual: { value: 'LessEqual', text: '<=' },
    GreaterEqual: { value: 'GreaterEqual', text: '>=' },
    ...NIL_OPERATORS,
};

const OPERATORS_ARRAY = {
    In: { value: 'In', text: 'Содержит' },
    NotIn: { value: 'NotIn', text: 'Не содержит' },
    ...NIL_OPERATORS,
};

export class SimpleJSONEQueryOperand extends React.Component<any, any> {

    state = {
        isCustomField: false,
        field: '',
        type: '',
        value: '',
    };

    componentDidMount(): void {
        const { field = '', type = '', value = '', fields } = this.props;
        const isCustomField = !Object.keys(fields).includes(field) && field !== '';
        this.setState({ field, type, value, isCustomField });
    }

    UNSAFE_componentWillUpdate(nextProps: Readonly<any>, nextState: Readonly<any>, nextContext: any): void {
        const { field, type, value } = this.props;
        const { field: fieldNext, type: typeNext, value: valueNext } = nextProps;

        if (field !== fieldNext || type !== typeNext || value !== valueNext) {
            const { field, type, value } = nextProps;
            this.setState({ field, type, value });
        }
    }

    onChange(type: string, valueType: string, value: any) {
        const data: any = {};

        if (valueType === TIMESTAMP) {
            value = +value / ONE_SECOND;
        }

        if (type === FIELD) {
            data.type = '';
            data.value = '';
        }

        const fieldType = type === FIELD && this.props.fields && type && this.props.fields[value]
            ? this.props.fields[value].Type
            : null;

        if (fieldType === BOOLEAN) {
            data.type = OPERATORS.Equal.value;
            data.value = false;
        }

        data[type] = value;

        this.setState(data, () => {
            const { field, type, value } = this.state;
            this.props.onChange({ field, type, value });
        });
    }

    handleCheckboxChange(isCustom) {
        this.setState({ isCustomField: isCustom });
    }

    render() {
        const { isCustomField, field, value, type } = this.state;
        const fieldsItems = this.props.fields && Object.entries(this.props.fields)
            ? Object.entries(this.props.fields).map((entry: [string, IOption]) => {
                const [key, value] = entry;

                return {
                    text: value.Title || key,
                    value: key,
                };
            })
            : [];

        let typeItems: any = [];

        let valueInput: any = null;
        const fieldType = this.props.fields && field && this.props.fields[field]
            ? this.props.fields[field].Type
            : null;
        let valueType = fieldType;

        if (fieldType === ARRAY) {
            valueType = this.props.fields[field].SubType && this.props.fields[field].SubType.Type;
        }

        switch (valueType) {
        case INTEGER:
            valueInput = <Input value={value}
                                disabled={this.props.disabled}
                                placeholder={'Значение'}
                                type={'number'}
                                className={`${style.json_query_input} ${style.value_input}`}
                                onChange={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        case TIMESTAMP:
            valueInput = <DatePicker value={+value * ONE_SECOND}
                                     disabled={this.props.disabled}
                                     className={style.value_input}
                                     placeholder={'Дата'}
                                     onChange={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        case FLOAT:
            valueInput = <Input value={value}
                                disabled={this.props.disabled}
                                placeholder={'Значение'}
                                type={'number'}
                                className={`${style.json_query_input} ${style.value_input}`}
                                onChange={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        case ENUM:
            const valueItems = this.props.fields[field].Values
                    && Object.entries(this.props.fields[field].Values)
                        .map((entry: [string, any]) => {
                            const [key, value] = entry;
                            const title = value.Title || key;

                            return {
                                text: title,
                                value: key,
                            };
                        });
            valueInput = <Select options={valueItems}
                                 initialValues={value ? [value] : []}
                                 disabled={this.props.disabled}
                                 placeholder={'Значение'}
                                 containerClassName={style.json_query_input}
                                 onSelect={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        case STRING:
            valueInput = <Input value={value}
                                disabled={this.props.disabled}
                                placeholder={'Значение'}
                                className={style.json_query_input}
                                onChange={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        case BOOLEAN:
            valueInput = <Checkbox checked={!!value}
                                   disabled={this.props.disabled}
                                   className={style.json_query_input}
                                   onChange={this.onChange.bind(this, 'value', valueType)}/>;
            break;
        default:
            valueInput = <Input value={value}
                                disabled={this.props.disabled}
                                placeholder={'Значение'}
                                className={style.json_query_input}
                                onChange={this.onChange.bind(this, 'value', valueType)}/>;
        }

        switch (fieldType) {
        case ARRAY:
            typeItems = OPERATORS_ARRAY
                    && Object.values(OPERATORS_ARRAY).map(operator => {
                        return {
                            text: operator.text,
                            value: operator.value,
                        };
                    });
            break;
        default:
            typeItems = OPERATORS
                    && Object.values(OPERATORS).map(operator => {
                        return {
                            text: operator.text,
                            value: operator.value,
                        };
                    });
        }

        return <div className={style.simple_json_operand}>
            <div data-tip data-for={'tag-index'}>
                <Checkbox className={style.json_query_input}
                          disabled={this.props.disabled}
                          checked={isCustomField}
                          onChange={this.handleCheckboxChange.bind(this)}/>
            </div>
            <ReactTooltip id={'tag-index'} aria-haspopup='true' type="info" effect="solid" children={'Custom column'}/>
            {isCustomField
                ? <Input className={`${style.json_query_input} ${style.variable_input}`}
                         disabled={this.props.disabled}
                         placeholder={'Поле'}
                         value={field ? field : ''}
                         onChange={this.onChange.bind(this, 'field', null)}/>
                : <Select options={fieldsItems}
                          initialValues={field ? [field] : []}
                          disabled={this.props.disabled}
                          placeholder={'Поле'}
                          containerClassName={`${style.json_query_input} ${style.variable_input}`}
                          onSelect={this.onChange.bind(this, 'field', null)}/>}
            {fieldType !== BOOLEAN
                ? <Select options={typeItems}
                          initialValues={type ? [type] : []}
                          disabled={this.props.disabled}
                          placeholder={'Оператор'}
                          containerClassName={`${style.json_query_input} ${style.operator_select}`}
                          onSelect={this.onChange.bind(this, 'type', null)}/>
                : null}
            {!Object.entries(NIL_OPERATORS).map(item => item[1].value).includes(type) && valueInput}
        </div>;
    }
}
