import React from 'react';

import CopyIcon from '../../../../../svg-components/copy.component.svg';
import FolderIcon from '../../../../../svg-components/folder.component.svg';
import KeyIcon from '../../../../../svg-components/key.component.svg';
import ResourceIcon from '../../../../../svg-components/resource.component.svg';
import SettingsIcon from '../../../../../svg-components/settings.component.svg';
import { Button } from '../../../../ui/Button';
import { Link } from '../../../../ui/Link';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { Popup } from '../../ui/Popup';
import { CORE_DRIVE_LENS_REQUESTS, REQUESTS } from '../request';
import { IAction, IPlanner, ISecret, OBJECTS_ID_PARAM } from '../types';
import styles from './index.css';

const menu = [
    { icon: 'folder', key: 'folder', name: 'Директория', disabled: true },
    { icon: 'copy', key: 'action', name: 'Действие', disabled: true },
    { icon: 'settings', key: 'config', name: 'Конфигурация', disabled: true },
    { icon: 'key', key: OBJECTS_ID_PARAM?.secret, name: 'Секрет', disabled: false },
    { icon: 'resource', key: 'file', name: 'Файл', disabled: true },
];

const generateIcon = (type, params) => {
    switch(type) {
    case 'folder':
        return <FolderIcon {...params}/>;
    case 'copy':
        return <CopyIcon {...params}/>;
    case 'settings':
        return <SettingsIcon {...params}/>;
    case 'key':
        return <KeyIcon {...params}/>;
    case 'resource':
        return <ResourceIcon {...params}/>;
    default:
        return null;
    }
};

interface IDir {
    CreateTime: number;
    Description: string;
    DirID: number;
    ID: number;
    Title: string;
}

interface INavigationList {
    Actions: IAction[];
    Configs: any[];
    CreateTime: number;
    DirID: number;
    Dirs: IDir[];
    ID: number;
    Planners: IPlanner[];
    Resources: any[];
    Secrets: ISecret[];
    Title: string;
}

interface INavigationProps {
    className?: string;
    deniedRootID?: number;
    dirID: number | string | null;
    handleObjectClick: () => void;
}

interface INavigationState {
    isLoading: boolean;
    isOpenPopup: boolean;
    navigationList: INavigationList | any;
    error: Error | null;
}

export class Navigation extends React.PureComponent<INavigationProps, INavigationState> {
    request = new Request2({ requestConfigs: CORE_DRIVE_LENS_REQUESTS });

    state: INavigationState = {
        isLoading: false,
        isOpenPopup: false,
        navigationList: {},
        error: null,
    };

    componentDidMount(): void {
        this.props.dirID && this.getDir(+this.props.dirID);
    }

    componentDidUpdate(prevProps: Readonly<INavigationProps>): void {
        if (this.props.dirID !== prevProps.dirID) {
            this.props.dirID && this.getDir(+this.props.dirID);
        }
    }

    getDir(id: number) {
        this.setState({ isLoading: true });
        this.request.exec(REQUESTS.GET_DIR, { queryParams: { id }, headers: { 'Cache-Control': 'no-cache' } })
            .then(response => {
                this.setState({ navigationList: response, isLoading: false });
            })
            .catch((error: Error) => {
                this.setState({ error, isLoading: false });
            });
    }

    handleClickMenu(key) {
        location.href = `#/analytics/tree?dirID=${this.props.dirID}&${key}=new`;
        this.togglePopup(false);
    }

    togglePopup(state) {
        this.setState({ isOpenPopup: state });
    }

    render() {
        const { isOpenPopup, isLoading, navigationList, error } = this.state;
        const { className, handleObjectClick, deniedRootID } = this.props;
        const { Actions, Planners, Secrets, Configs, Resources } = navigationList;
        const navigationItems = [
            ...(Actions || []).map(action => ({ ...action, type: 'action' })),
            ...(Planners || []).map(planner => ({ ...planner, type: 'planner' })),
            ...(Secrets || []).map(secret => ({ ...secret, type: 'secret' })),
            ...(Configs || []).map(config => ({ ...config, type: 'config' })),
            ...(Resources || []).map(resource => ({ ...resource, type: 'resource' })),
        ].sort((a, b) => a.Title.localeCompare(b.Title));

        return <div className={`${styles.navigation} ` + (className || '')}>
            {error
                ? <SimpleError error={error}/>
                : <>
                    <div className={styles['navigation__button-container']}>
                        <Button onClick={this.togglePopup.bind(this, true)}
                                basic={true}>Создать</Button>
                        <Popup isOpen={isOpenPopup} togglePopup={this.togglePopup.bind(this)}>
                            {menu.map(item =>
                                <div key={item.name}
                                     onClick={this.handleClickMenu.bind(this, item.key)}
                                     className={`${styles.menu__item} ${(item.disabled && styles.menu__item_disabled)}`}>
                                    { generateIcon(item.icon, { className: styles.menu__icon }) }
                                    <span>{item.name}</span>
                                </div>)}
                        </Popup>
                    </div>
                    {isLoading
                        ? <Spin size={'l'}/>
                        : <>
                            {navigationList.DirID && deniedRootID !== navigationList.DirID &&
                            <div className={styles.navigation__item}>
                                <div className={`${styles.navigation__icon} ${styles.navigation__icon_dir}`}>D</div>
                                <Link onClick={handleObjectClick.bind(null, OBJECTS_ID_PARAM.dir, navigationList.DirID)}
                                      children={'...'}/>
                            </div>}
                            <ul className={styles.navigation__list}>
                                {navigationList?.Dirs
                                    ?.sort?.((a, b) => a.Title.localeCompare(b.Title))
                                    .map((dir, index) =>
                                        <li key={index} className={styles.navigation__item}>
                                            <div className={`${styles.navigation__icon} ${styles.navigation__icon_dir}`}>D
                                            </div>
                                            <Link onClick={handleObjectClick.bind(null, OBJECTS_ID_PARAM.dir, dir.ID)}
                                                  children={dir.Title}/>
                                        </li>,
                                    )}
                                {navigationItems.map((item, index) => {
                                    return <li key={item.Title + index} className={styles.navigation__item}>
                                        <div className={`${styles.navigation__icon} ${styles['navigation__icon_' + item.type]}`}
                                             children={item.type[0].toUpperCase()}/>
                                        {
                                            ['action', 'runAction', 'task', 'secret'].includes(item?.type)
                                                ? <Link title={item.Title}
                                                        onClick={handleObjectClick
                                                            .bind(null, OBJECTS_ID_PARAM[item.type], item.ID)}
                                                        className={styles.navigation__link}
                                                        children={item.Title}/>
                                                : <Link title={item.Title}
                                                        disable={true}
                                                        className={styles.navigation__link}
                                                        children={item.Title}/>
                                        }
                                    </li>;
                                })}
                            </ul>
                        </>}
                </>}
        </div>;
    }
}
