import React, { memo,useEffect, useState } from 'react';

import { Button, ButtonTypes } from '../../../../../ui/Button';
import { Card } from '../../../../../ui/Card';
import { Collapse } from '../../../../../ui/Collapse';
import { Confirm } from '../../../../../ui/FullModal';
import { Input } from '../../../../../ui/Input';
import { Request2 } from '../../../../../utils/request';
import { SimpleError } from '../../../../SimpleError';
import { CORE_DRIVE_LENS_REQUESTS,REQUESTS } from '../../request';
import styles from './index.css';

interface ISecret {
    ID?: number | undefined;
    DirID: number;
    Title: string;
    Type: string;
    Data: {
        SecretID: string;
        VersionID: string;
        Key: string;
    };
}

interface IRequestInfo {
    error: Error | null;
    isLoading: boolean;
    isDeleting: boolean;
}

interface ISecretViewProps {
    dirID: number | undefined;
    secretID: string;
    className: string;
    setDir: (dirID: number) => void;
}

export const SecretView = memo((props: ISecretViewProps) => {
    const request = new Request2({ requestConfigs: CORE_DRIVE_LENS_REQUESTS });
    const {
        dirID,
        secretID,
        className,
        setDir,
    } = props;
    const [isConfirmOpen, setConfirmOpen] = useState(false);
    const [requestInfo, setRequestInfo] = useState<IRequestInfo>({
        isLoading: false,
        isDeleting: false,
        error: null,
    });
    const [data, setData] = useState<ISecret>({
        DirID: dirID && dirID || 1,
        Title: '',
        Type: 'Yav',
        Data: {
            SecretID: '',
            VersionID: '',
            Key: '',
        },
    });

    useEffect(() => {
        secretID === 'new'
            ? setData({ ...data, Title: '', Data: { SecretID: '', VersionID: '', Key: '' } })
            : getSecret(secretID);

        return () => {
            request.abort();
        };
    }, [secretID]);

    const createSecret = () => {
        setRequestInfo({ ...requestInfo, isLoading: true });
        request.exec(REQUESTS.CREATE_SECRET, { body: data })
            .then((secret: ISecret) => {
                setRequestInfo({ ...requestInfo, isLoading: false });
                location.href = `#/analytics/tree?secretID=${secret.ID}`;
            })
            .catch(error => {
                setRequestInfo({ ...requestInfo, isLoading: false, error });
            });
    };

    const getSecret = (id: string) => {
        setRequestInfo({ ...requestInfo, isLoading: true });
        request.exec(REQUESTS.GET_SECRET, { queryParams: { id }, headers: { 'Cache-Control': 'no-cache' } })
            .then(secret => {
                setRequestInfo({ ...requestInfo, isLoading: false, error: null });
                setData(secret);
                setDir(secret.DirID);
            })
            .catch(error => {
                setRequestInfo({ ...requestInfo, isLoading: false, error });
            });
    };

    const updateSecret = (id: string) => {
        setRequestInfo({ ...requestInfo, isLoading: true });
        request.exec(REQUESTS.UPDATE_SECRET,
            { queryParams: { id }, body: data })
            .then(secret => {
                setRequestInfo({ ...requestInfo, isLoading: false, error: null });
                setData(secret);
                setDir(secret.DirID);
            })
            .catch(error => {
                setRequestInfo({ ...requestInfo, isLoading: false, error });
            });
    };

    const deleteSecret = (id: string) => {
        setRequestInfo({ ...requestInfo, isDeleting: true });
        request.exec(REQUESTS.DELETE_SECRET, { queryParams: { id } })
            .then(() => {
                setRequestInfo({ ...requestInfo, isDeleting: false, error: null });
                setConfirmOpen(false);
                location.href = `#/analytics/tree?dirID=${data.DirID}`;
            })
            .catch(error => {
                setRequestInfo({ ...requestInfo, isDeleting: false, error });
                setConfirmOpen(false);
            });
    };

    return <div className={className ? className : ''}>
        <Card title={secretID === 'new' ? 'Создание секрета' : 'Редактирование секрета'}>
            {requestInfo.error
                ? <SimpleError error={requestInfo.error}/>
                : <>
                    <Input placeholder={'Имя'}
                           value={data.Title}
                           disabled={requestInfo.isLoading}
                           onChange={(value) => setData({ ...data, Title: value })}/>
                    <Input placeholder={'SecretID'}
                           value={data.Data.SecretID}
                           disabled={requestInfo.isLoading}
                           onChange={(value) => setData({ ...data, Data: { ...data.Data, SecretID: value } })}/>
                    <Input placeholder={'Ключ'}
                           value={data.Data.Key}
                           disabled={requestInfo.isLoading}
                           onChange={(value) => setData({ ...data, Data: { ...data.Data, Key: value } })}/>
                    <Collapse title={'Дополнительно'} collapsed={true} className={styles.secret__collapse}>
                        <Input placeholder={'VersionID'}
                               value={data.Data.VersionID}
                               disabled={requestInfo.isLoading}
                               onChange={(value) => setData({ ...data, Data: { ...data.Data, VersionID: value } })}/>
                    </Collapse>
                    {secretID === 'new'
                        ? <Button className={`${styles.secret__button} ${styles.secret__button_create}`}
                                  isLoading={requestInfo.isLoading}
                                  disabled={requestInfo.isDeleting}
                                  onClick={() => createSecret()}>Создать</Button>
                        : <Button className={`${styles.secret__button} ${styles.secret__button_save}`}
                                  isLoading={requestInfo.isLoading}
                                  disabled={requestInfo.isDeleting}
                                  onClick={() => updateSecret(secretID)}>Сохранить</Button>
                    }
                    <Button className={styles.secret__button}
                            colorType={ButtonTypes.negative}
                            basic={true}
                            disabled={requestInfo.isLoading}
                            isLoading={requestInfo.isDeleting}
                            onClick={() => setConfirmOpen(true)}>Удалить</Button>
                </>}
        </Card>
        {isConfirmOpen &&
        <Confirm question={'Вы действительно хотите удалить?'}
                 error={requestInfo.error}
                 isWorking={requestInfo.isDeleting}
                 accept={() => deleteSecret(secretID)}
                 onClose={() => setConfirmOpen(false)}/>}
    </div>;
});
