import React from 'react';

import { ONE_DAY, ONE_SECOND } from '../../../../constants';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { Card } from '../../../../ui/Card';
import DatePicker from '../../../../ui/DatePicker';
import { Confirm, Window } from '../../../../ui/FullModal';
import { Input } from '../../../../ui/Input';
import Select from '../../../../ui/Select';
import { Tabs } from '../../../../ui/Tabs';
import TextArea from '../../../../ui/TextArea';
import { isValidJSONString } from '../../../../utils/isValidJSONString';
import { Request2 } from '../../../../utils/request';
import { ErrorSource, logError } from '../../../Content/initErrorCounter';
import { SimpleError } from '../../../SimpleError';
import { getTranslateKey, PROGRESS_STATUSES } from '../constants';
import { LogicalJSONEditor } from '../LogicalJSONEditor/component';
import { CORE_DRIVE_LENS_REQUESTS, REQUESTS } from '../request';
import { Resources } from '../Resources/component';
import { IAction, IOption, IResource, ITask, OBJECTS_ID_PARAM } from '../types';
import styles from './index.css';

const JSON_QUERY = 'JSONQuery';
const STRING = 'String';
const INTEGER = 'Integer';
const SECRET = 'Secret';
const DATE = 'Date';
const DATE_TIME = 'DateTime';

const DELAY_IN_SEC = 5;
const DELAY = ONE_SECOND * DELAY_IN_SEC;

enum LogTypes {
    OUT = 'stdout',
    ERROR = 'stderr',
    SYSTEM = 'system'
}

interface ISecretBuildControl {
    title: string;
    values: {};
    key: string;
    editable: boolean;
    type: string;
}

interface IBuildControl {
    title: string;
    values: {};
    key: string;
    editable: boolean;
    type: string;
    settings: any;
}

interface IObjectViewProps {
    handleObjectClick: (type: string, id) => void;
    setInfoFromAction?: (dirID: number | null, tasks: ITask[]) => void;
    getTasksList?: () => void;
    actionID: string | null;
    runActionID: string | null;
    taskID: string | null;
    t: any;
}

interface IObjectViewState {
    action: IAction;
    task: ITask;
    visibleOptions: string[];
    values: { [key: string]: any };
    isAbortConfirmOpen: boolean;
    abortingError: Error | null;
    logError: Error | null;
    error: Error | null;
    isLogModalOpen: boolean;
    log: string;
    currentTab: string;
    isProgress: boolean;
    resources: IResource[];
}

export class ObjectView extends React.Component<IObjectViewProps, IObjectViewState> {
    request = new Request2({ requestConfigs: CORE_DRIVE_LENS_REQUESTS });
    timer: any;
    tabs = Object.keys(LogTypes).map(name => ({ name, link: LogTypes[name] }));
    state: IObjectViewState = {
        action: {} as IAction,
        task: {} as ITask,
        visibleOptions: [],
        values: {},
        isAbortConfirmOpen: false,
        abortingError: null,
        logError: null,
        error: null,
        isLogModalOpen: false,
        log: '',
        currentTab: this.tabs[0].name,
        isProgress: false,
        resources: [],
    };

    componentDidMount(): void {
        const { actionID, runActionID, taskID } = this.props;
        this.setAction(actionID || runActionID, null);
        this.setTask(taskID);
    }

    componentDidUpdate(prevProps: Readonly<IObjectViewProps>): void {
        if (this.props.actionID !== prevProps.actionID) {
            const { actionID } = this.props;
            this.setAction(actionID, null);
        }

        if (this.props.taskID !== prevProps.taskID) {
            const { taskID } = this.props;
            this.setTask(taskID);
        }

        if (this.props.runActionID !== prevProps.runActionID) {
            const { runActionID } = this.props;
            this.setAction(runActionID, this.state.values);
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
        clearTimeout(this.timer);
    }

    setAction(actionID: string | null, valueFromTask: { [key: string]: any } | null) {
        this.getAction(actionID, valueFromTask).then((action: IAction) => {
            this.props.setInfoFromAction && this.props.setInfoFromAction(action.DirID, action.Tasks || []);
        });
    }

    getAction(actionID: string | null, valueFromTask: { [key: string]: any } | null) {
        return new Promise(resolve => {
            clearTimeout(this.timer);
            actionID && this.request.exec(REQUESTS.GET_ACTION, { queryParams: { id: actionID } })
                .then(action => {
                    const visibleOptions = this.getVisibleOptions(action.Options);
                    let values = {};
                    visibleOptions.forEach(key => values[key] = action.Options[key].Value);
                    const isProgress = action.Tasks.some(task => PROGRESS_STATUSES.includes(task.Status));
                    values = valueFromTask && valueFromTask || values;
                    this.setState({ action, visibleOptions, values, error: null }, () => {
                        resolve(action);
                        this.props.setInfoFromAction && this.props.setInfoFromAction(null, action.Tasks);
                        isProgress && !this.props.runActionID && (this.timer = setTimeout(
                            () => this.getAction(actionID, null), DELAY,
                        )
                        );
                    });
                }).catch(error => this.setState({ error }));
        });

    }

    copyActionFromTask(task: ITask) {
        const { Action: action, Options: options } = task;
        const values = {};
        const mergeOptions = { ...action.Options, ...options };
        const visibleOptions = this.getVisibleOptions(task.Action.Options);
        visibleOptions.forEach(value => {
            values[value] = mergeOptions[value].Value;
        });
        this.setState({ action, values, visibleOptions }, () => {
            this.props.handleObjectClick(OBJECTS_ID_PARAM.runAction, task.ActionID);
        });
    }

    setTask(taskID: string | null) {
        this.getTask(taskID).then((task: ITask) => {
            this.props.setInfoFromAction && this.props.setInfoFromAction(task.Action.DirID, task.Action.Tasks || []);
        });
    }

    getTask(taskID: string | null) {
        return new Promise(resolve => {
            clearTimeout(this.timer);
            taskID && this.request.exec(REQUESTS.GET_TASK, { queryParams: { id: taskID } })
                .then(task => {
                    const { Action: action, Options: options, Status: status, Resources: resources = [] } = task;
                    const values = {};
                    const mergeOptions = { ...action.Options, ...options };
                    const visibleOptions = this.getVisibleOptions(task.Action.Options);
                    const isProgress = PROGRESS_STATUSES.includes(status);
                    visibleOptions.forEach(value => {
                        values[value] = mergeOptions[value].Value;
                    });
                    this.setState({
                        task, action: task.Action, values, visibleOptions, isProgress,
                        resources, error: null,
                    }, () => {
                        resolve(task);
                        this.props.setInfoFromAction && this.props.setInfoFromAction(null, task.Action.Tasks);
                        isProgress && (this.timer = setTimeout(() => this.getTask(taskID), DELAY));
                    });
                }).catch(error => this.setState({ error }));

        });
    }

    getVisibleOptions(options) {
        return options && Object.keys(options).filter(key => options[key].Visible) || [];
    }

    runTask() {
        const values = Object.assign({}, this.state.values);
        const { visibleOptions, action } = this.state;
        visibleOptions.forEach((key: string) => {
            const { Type: type } = this.state.action.Options[key];
            if (type === JSON_QUERY) {
                values[key].Operands.forEach(operand => {
                    const isCustom = !Object.keys(action.Options.Query.Settings.Fields).includes(operand.Field);
                    operand.Value = (isCustom && +operand.Value || +operand.Value === 0)
                        ? +operand.Value
                        : operand.Value;
                });
                values[key] = isValidJSONString(values[key]) ? JSON.parse(values[key]) : values[key];
            }

            if (type === INTEGER) {
                values[key] = +values[key];
            }

            values[key] = {
                Type: type,
                Value: values[key],
            };
        });

        const requestData = { Options: values };
        this.request.exec(REQUESTS.RUN_TASK, { queryParams: { id: this.state.action.ID }, body: requestData })
            .then((task) => {
                this.setState({ task }, () => {
                    this.props.handleObjectClick(OBJECTS_ID_PARAM.task, task.ID);
                    this.props.getTasksList && this.props.getTasksList();
                });
            });
    }

    openAbortConfirm() {
        this.setState({ isAbortConfirmOpen: true });
    }

    closeAbortConfirm() {
        this.setState({ isAbortConfirmOpen: false });
    }

    openLogModal() {
        this.setState({ isLogModalOpen: true }, () => {
            this.selectTab(this.tabs[0].link);
        });
    }

    selectTab(logType: string) {
        this.request.exec(REQUESTS.GET_TASK_LOG, {
            queryParams: {
                id: this.props.taskID,
                logType: logType,
            },
        }).then(log => {
            this.setState({ currentTab: logType, log, logError: null });
        }).catch(error => this.setState({ logError: error }));
    }

    closeLogModal() {
        this.setState({ isLogModalOpen: false });
    }

    abortTask() {
        this.request.exec(REQUESTS.ABORT_TASK, { queryParams: { id: this.state.task.ID } })
            .then((task) => {
                this.setState({ isAbortConfirmOpen: false, abortingError: null });
                this.props.handleObjectClick(OBJECTS_ID_PARAM.action, task.ActionID);
            })
            .catch((abortingError: Error) => {
                this.setState({ abortingError });
            });
    }

    onValueChange(key: string, type: string, value: any) {
        const values = this.state.values;
        values[key] = value;
        this.setState({ values });
    }

    handleDateChange(key: string, type: string, value: number, onlyDate: boolean) {
        value = onlyDate
            ? Math.trunc(value / (ONE_DAY * ONE_SECOND)) * (ONE_DAY * ONE_SECOND)
            : Math.trunc(value / ONE_SECOND);
        this.onValueChange.call(this, key, type, value);
    }

    handleDateStringChange(key: string, type: string, value: number) {
        const DATE_RANGE_LIMIT = 10;
        this.onValueChange.call(this, key, type, new Date(value).toISOString().slice(0, DATE_RANGE_LIMIT));
    }

    handleSecretChange(key: string, type: string, field: string, value: number) {
        this.onValueChange.call(this, key, type, { ...this.state.values[key], [field]: value });
    }

    secretBuildControl(parameters: ISecretBuildControl) {
        const { values, key, editable, type } = parameters;
        const input = <>
            {values[key]?.Store && <Select initialValues={[values[key]?.Store]}
                                           placeholder={'Store'}
                                           disabled
                                           options={[{ value: values[key]?.Store }]}
                                           onSelect={this.handleSecretChange.bind(this)}/>}
            <Input placeholder={'Name'}
                   value={values[key]?.Name}
                   disabled={!editable}
                   onChange={this.handleSecretChange.bind(this, key, type, 'Name')}/>
        </>;

        return input;
    }

    integerBuildControl(parameters: IBuildControl) {
        const { title, values, key, editable, type, settings } = parameters;
        const displayTime = settings && settings.DisplayType;
        const relativeTime = settings && settings.RelativeTime || null;
        let input: React.ReactNode;
        if (editable && !values[key] && relativeTime) {
            values[key] = Math.trunc(Date.now() / ONE_SECOND) + relativeTime;
        }

        switch (displayTime) {
        case DATE:
            input = <DatePicker utc
                                onlyDate={true}
                                placeholder={title + ' ' + 'UTC'}
                                disabled={!editable}
                                value={values[key] * ONE_SECOND}
                                onChange={this.handleDateChange.bind(this, key, type)}/>;
            break;
        case DATE_TIME:
            input = <DatePicker utc
                                placeholder={title + ' ' + 'UTC'}
                                value={values[key] * ONE_SECOND}
                                disabled={!editable}
                                onChange={this.handleDateChange.bind(this, key, type)}/>;
            break;
        default:
            input = <Input placeholder={title}
                           value={values[key]}
                           disabled={!editable}
                           type={'number'}
                           onChange={this.onValueChange.bind(this, key, type)}/>;
        }

        return input;
    }

    stringBuildControl(parameters: IBuildControl) {
        const { title, values, key, editable, type, settings } = parameters;
        const displayTime = settings && settings.DisplayType;
        const relativeTime = settings && settings.RelativeTime || null;
        let input: React.ReactNode;
        if (editable && !values[key] && relativeTime !== null) {
            values[key] = Math.trunc(Date.now() / ONE_SECOND) + relativeTime;
        }

        switch (displayTime) {
        case DATE:
            input = <DatePicker utc
                                onlyDate={true}
                                placeholder={title + ' ' + 'UTC'}
                                value={Date.parse(values[key])}
                                disabled={!editable}
                                onChange={this.handleDateStringChange.bind(this, key, type)}/>;
            break;
        case DATE_TIME:
            input = <DatePicker utc
                                placeholder={title + ' ' + 'UTC'}
                                value={Date.parse(values[key])}
                                disabled={!editable}
                                onChange={this.handleDateStringChange.bind(this, key, type)}/>;
            break;
        default:
            input = settings && settings.Multiline
                ? <TextArea placeholder={title}
                            value={values[key]}
                            disabled={!editable}
                            onChange={this.onValueChange.bind(this, key, type)}/>
                : <Input placeholder={title}
                         value={values[key]}
                         onChange={this.onValueChange.bind(this, key, type)}
                         disabled={!editable}/>;
        }

        return input;
    }

    buildControl(key: string, option: IOption) {
        let input: React.ReactNode;
        let { Description: description, Type: type, Editable: editable, Title: title, Settings: settings } = option;
        !this.props.runActionID && (editable = false);
        const values: { [key: string]: any } = this.state.values;
        switch (type) {
        case JSON_QUERY:
            const fields = option && option.Settings && option.Settings.Fields && option.Settings.Fields || {};
            input = <LogicalJSONEditor onChange={this.onValueChange.bind(this, key, type)}
                                       fields={fields}
                                       disabled={!editable}
                                       json={values[key]}/>;
            break;
        case STRING:
            input = this.stringBuildControl({ title, values, key, editable, type, settings });
            break;
        case INTEGER:
            input = this.integerBuildControl({ title, values, key, editable, type, settings });
            break;
        case SECRET:
            input = this.secretBuildControl({ title, values, key, editable, type });
            break;
        default:
            input = <Input placeholder={title}
                           value={values[key]}
                           disabled={!editable}
                           onChange={this.onValueChange.bind(this, key, type)}/>;
        }

        return <div key={key}>
            {input}
            <span className={styles.option_description}>{description}</span>
        </div>;
    }

    buildOptionRow(key: string, option: IOption) {
        return <tr key={key}>
            <td>{option.Title}</td>
            <td>{this.buildControl(key, option)}</td>
        </tr>;
    }

    actionButtons() {
        return <>
            <Button className={styles.object__button}
                    onClick={this.props.handleObjectClick.bind(null, OBJECTS_ID_PARAM.runAction, this.props.actionID)}
                    colorType={ButtonTypes.positive}
                    basic>
                {this.props.t.getItem(getTranslateKey('create'))}
            </Button>
            <Button className={styles.object__button}
                    disabled={true}>
                {this.props.t.getItem(getTranslateKey('plan'))}</Button>
            <Button className={styles.object__button}
                    disabled={true}>
                {this.props.t.getItem(getTranslateKey('move'))}
            </Button>
            <Button className={styles.object__button}
                    disabled={true}>{this.props.t.getItem(getTranslateKey('edit'))}</Button>
            <Button className={styles.object__button}
                    disabled={true}>
                {this.props.t.getItem(getTranslateKey('remove'))}
            </Button>
        </>;
    }

    runActionButtons() {
        return <>
            <Button colorType={ButtonTypes.positive}
                    className={styles.object__button}
                    onClick={this.runTask.bind(this)}>{this.props.t.getItem(getTranslateKey('run'))}</Button>
            <Button basic
                    colorType={ButtonTypes.warning}
                    className={styles.object__button}
                    onClick={this.props.handleObjectClick.bind(null, OBJECTS_ID_PARAM.action, this.props.runActionID)}>
                {this.props.t.getItem(getTranslateKey('return'))}
            </Button>
        </>;
    }

    taskButtons(isProgress: boolean) {
        return <>
            <Button basic
                    colorType={ButtonTypes.positive}
                    className={styles.object__button}
                    onClick={this.copyActionFromTask.bind(this, this.state.task)}>
                {this.props.t.getItem(getTranslateKey('clone'))}
            </Button>
            <Button basic
                    colorType={ButtonTypes.positive}
                    className={styles.object__button}
                    onClick={this.openLogModal.bind(this)}>
                {this.props.t.getItem(getTranslateKey('logs'))}
            </Button>
            {isProgress && <Button basic
                                   colorType={ButtonTypes.negative}
                                   onClick={this.openAbortConfirm.bind(this)}
                                   className={styles.object__button}>
                {this.props.t.getItem(getTranslateKey('abort'))}
            </Button>}
            {this.state.abortingError && <SimpleError error={this.state.abortingError}/>}
        </>;
    }

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({ error });
    }

    render() {
        const {
            action, task, visibleOptions, currentTab, log, isLogModalOpen, isAbortConfirmOpen, isProgress, resources,
            error, abortingError, logError,
        } = this.state;
        const { actionID, runActionID, taskID } = this.props;
        let title = action.ID ? `${action.Title} (ID: ${action.ID})` : '';
        title += (taskID && task.ID) ? ` (Task ID: ${task.ID}) ${task.Status}` : '';

        return <div className={styles.object}>
            {error ? <SimpleError error={error}/> : <Card title={title}>
                {action.Description && <div className={styles.object__description}>{action.Description}</div>}
                <table className={styles.object__options}>
                    <tbody>
                        {visibleOptions.map(key => this.buildOptionRow(key, action.Options[key]))}
                    </tbody>
                </table>
                {taskID && resources && <Resources resources={resources}/>}
                <div className={styles.object__controls}>
                    {actionID && this.actionButtons()}
                    {runActionID && this.runActionButtons()}
                    {taskID && this.taskButtons(isProgress)}
                </div>
            </Card>}
            {isLogModalOpen
                ?
                <Window className={styles.logs} error={logError} title={'Логи'} onClose={this.closeLogModal.bind(this)}>
                    <Tabs tabs={this.tabs} currentTab={currentTab} selectTab={this.selectTab.bind(this)}/>
                    {log && <pre className={styles.logs__log}>{log}</pre>}
                </Window>
                : null}
            {isAbortConfirmOpen
                ? <Confirm question={this.props.t.getItem(getTranslateKey('abort_confirm'))}
                           error={abortingError}
                           accept={this.abortTask.bind(this)}
                           isWorking={false}
                           onClose={this.closeAbortConfirm.bind(this)}/>
                : null}
        </div>;
    }
}
