import moment from 'moment';
import React from 'react';

import { EMPTY_DATA, LOCALIZATION_RESOURCE_PREFIX, ONE_SECOND } from '../../../../constants';
import { Card } from '../../../../ui/Card';
import FormatDate from '../../../../ui/FormatDate';
import { Link } from '../../../../ui/Link';
import { LabelStatus, TLabel } from '../../../../ui/Table';
import styleTable from '../../../../ui/Table/index.css';
import { Translate } from '../../../../utils/translate';
import { pad } from '../../../../utils/utils';
import { ITask, OBJECTS_ID_PARAM } from '../types';
import styles from './index.css';

interface ITasksTableProps {
    t: Translate;
    tasks: ITask[];
    previousIndex?: number;
    handleObjectClick: (type: string, id) => void;
}

enum taskStatuses {
    RUNNING = 'running',
    QUEUED = 'queued',
    FAILED = 'failed',
    SUCCEEDED = 'succeeded'
}

const TRANSLATE_NAMESPACE = `${LOCALIZATION_RESOURCE_PREFIX}analytics_tree_`;
const getTranslateKey = (key: string) => `${TRANSLATE_NAMESPACE}${key}`;

export class TaskHistory extends React.Component<ITasksTableProps> {

    render() {
        const { previousIndex = 0 } = this.props;
        const TASK_DATE_LIMIT = 8;
        const title = `${this.props.t.getItem(getTranslateKey('tasks_title'))}` +
            ` - последнее обновление: ${new Date().toTimeString().slice(0, TASK_DATE_LIMIT)}`;

        return <Card title={title}>
            <table className={styleTable.table}>
                <thead>
                    <tr>
                        <td>#</td>
                        <td>Статус</td>
                        <td>ID</td>
                        <td>Время запуска</td>
                        <td>Комментарий</td>
                    </tr>
                </thead>
                <tbody>
                    {this.props.tasks.map((task: ITask, index) => {
                        const { ID: id, Status: taskStatus } = task;
                        let statusType = '';
                        switch (taskStatus.toLowerCase()) {
                        case taskStatuses.RUNNING:
                            statusType = 'WARNING';
                            break;
                        case taskStatuses.QUEUED:
                            statusType = 'WARNING';
                            break;
                        case taskStatuses.FAILED:
                            statusType = 'NEGATIVE';
                            break;
                        case taskStatuses.SUCCEEDED:
                            statusType = 'POSITIVE';
                            break;
                        default:
                            statusType = 'DEFAULT';
                        }

                        const createTime = task.CreateTime ? moment(task.CreateTime * ONE_SECOND) : null;
                        const updateTime = task.UpdateTime ? moment(task.UpdateTime * ONE_SECOND) : null;
                        const diff = createTime && updateTime
                            ? updateTime.diff(createTime)
                            : null;
                        const duration = diff ? moment.duration(diff) : null;
                        const hours = duration && duration.hours() || 0;
                        const minutes = duration && duration.minutes() || 0;
                        const seconds = duration && duration.seconds() || 0;
                        const PAD_LENGTH= 2;
                        const hoursString = pad(hours, PAD_LENGTH);
                        const minutesString = pad(minutes, PAD_LENGTH);
                        const secondsString = pad(seconds, PAD_LENGTH);
                        const durationString = duration ? `${hoursString}:${minutesString}:${secondsString}` : null;

                        return <tr key={id}>
                            <td>{previousIndex + index + 1}</td>
                            <td>
                                <TLabel text={taskStatus} status={LabelStatus[statusType]}/>
                            </td>
                            <td>
                                <Link onClick={this.props.handleObjectClick
                                    .bind(this, OBJECTS_ID_PARAM.task, task.ID) || null}>
                                    {id || EMPTY_DATA}
                                </Link>
                            </td>
                            <td>
                                <div>
                                    <div>
                                        <FormatDate value={task.CreateTime * ONE_SECOND} withSecond={true}/>
                                    </div>
                                    {task.CreateTime && task.UpdateTime
                                && (diff && task.Status === 'finished' || task.Status === 'failed')
                                        ? <div>({durationString})</div> : null}
                                </div>
                            </td>
                            <td className={styles.table__comment}>
                                {task.Options && task.Options.Comment && task.Options.Comment.Value || EMPTY_DATA}
                            </td>
                        </tr>;
                    })}
                </tbody>
            </table>
        </Card>;
    }
}
