import React from 'react';

import { Translate } from '../../../utils/translate';
import { IStore } from '../../App/store';
import styles from './index.css';
import { Navigation } from './Navigation/component';
import { ObjectView } from './ObjectView/component';
import { SecretView } from './ObjectView/SecretView/component';
import { TaskHistory } from './TaskHistory/component';
import { ITask, OBJECTS_ID_PARAM } from './types';

interface ITreeProps extends IStore {
    location: Location;
}

interface ITreeState {
    actionID: string | null;
    runActionID: string | null;
    taskID: string | null;
    secretID: string | null;
    dirID: number | string | null;
    tasks: ITask[];
}

export default class Tree extends React.Component<ITreeProps, ITreeState> {
    t = this.props.Lang && new Translate(this.props.Lang) || {} as Translate;

    constructor(props) {
        super(props);
        const { dirID, actionID, runActionID, taskID, secretID } = this.getObjectsIds();
        this.state = {
            actionID: actionID || null,
            runActionID: runActionID || null,
            taskID: taskID || null,
            secretID: secretID || null,
            dirID: dirID || ((actionID || runActionID || taskID || secretID) ? null : 1),
            tasks: [],
        };
    }

    getObjectsIds() {
        const searchParams = new URLSearchParams(this.props.location && this.props.location.search || ' ');

        const dirID = searchParams.get(OBJECTS_ID_PARAM.dir);

        const actionID = searchParams.get(OBJECTS_ID_PARAM.action);
        const runActionID = searchParams.get(OBJECTS_ID_PARAM.runAction);
        const taskID = searchParams.get(OBJECTS_ID_PARAM.task);
        const secretID = searchParams.get(OBJECTS_ID_PARAM.secret);

        return { dirID, actionID, runActionID, taskID, secretID };
    }

    componentDidUpdate(prevProps: Readonly<ITreeProps>, prevState: Readonly<{}>, snapshot?: any): void {
        if (this.props.location.search !== prevProps.location.search) {

            const { dirID, actionID, runActionID, taskID, secretID } = this.getObjectsIds();
            if (dirID) {
                this.setState({ dirID: +dirID, secretID });
            } else {
                this.setState({ actionID, runActionID, taskID, secretID });
            }
        }
    }

    componentWillUnmount(): void {
        const position = location.hash.indexOf('?');
        position !== -1 && (location.hash = location.hash.slice(0, position));
    }

    handleObjectClick(type: string, ID) {
        location.href = `#/analytics/tree?${type}=${ID}`;
    }

    setDir(dirID: number) {
        if (typeof this.state.dirID === 'string') {
            this.setState({ dirID: +dirID });
        } else {
            this.setState({ dirID: String(dirID) });
        }
    }

    setInfoFromAction(dirID: number | null, tasks: ITask[]) {
        dirID ? this.setState({ dirID, tasks }) : this.setState({ tasks });
    }

    render() {
        const { dirID, actionID, runActionID, taskID, secretID, tasks } = this.state;

        return <div className={styles.tree}>
            <Navigation className={styles.tree__navigation}
                        dirID={dirID}
                        handleObjectClick={this.handleObjectClick.bind(this)}/>
            {(actionID || runActionID || taskID || secretID) &&
            <div className={styles.tree__object}>
                {secretID
                    ? <SecretView className={styles.tree__secret}
                                  secretID={secretID}
                                  setDir={this.setDir.bind(this)}
                                  dirID={dirID && +dirID || 1}/>
                    : <ObjectView t={this.t}
                                  handleObjectClick={this.handleObjectClick.bind(this)}
                                  setInfoFromAction={this.setInfoFromAction.bind(this)}
                                  actionID={actionID}
                                  runActionID={runActionID}
                                  taskID={taskID}/>}
                {actionID && tasks &&
                <TaskHistory t={this.t} tasks={tasks} handleObjectClick={this.handleObjectClick.bind(this)}/>}
            </div>}
        </div>;
    }
}
