import React from 'react';

import { Button } from '../../../ui/Button';
import { Confirm, Window } from '../../../ui/FullModal';
import { JsonModal } from '../../../ui/FullModal/JsonModal';
import { Link } from '../../../ui/Link';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import TagsHistory from '../../Clients/UserInfoView/TagsHistory';
import { HISTORY_TYPES } from '../../Clients/UserInfoView/TagsHistory/component';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import TagModal from '../../TagModal';
import { OBJECT_TYPES, Operations } from '../../TagModal/component';
import { IWalletOrganization } from '../SearchWallets/types';
import style from './index.css';
import { requestConfigs, REQUESTS } from './request';

interface IAccountTagsModalProps {
    onClose: () => void;
    organization: IWalletOrganization;
}

const AccountTagsModalContext = React.createContext<IContext>({} as IContext);

interface IAccountTagsModalState {
    showTagDialog: boolean;
    listIsLoading: boolean;
    detailsIsLoading: boolean;
    tagsList: any[];
    listError: null | Error;
    showingTag: null | any;
    editingTag: null | any;
    deletingTag: null | any;
    deleteError: null | Error;
    errorDetails: null | Error;
    deleteIsWorking: boolean;
}

interface IContext extends IAccountTagsModalState {
    object_id: any;
}

export class AccountTagsModal extends React.Component<IAccountTagsModalProps, IAccountTagsModalState> {
    state: IAccountTagsModalState = {
        showTagDialog: false,
        listIsLoading: false,
        detailsIsLoading: false,
        tagsList: [],
        listError: null,
        showingTag: null,
        editingTag: null,
        deletingTag: null,
        deleteError: null,
        errorDetails: null,
        deleteIsWorking: false,
    };
    request = new Request2({ requestConfigs });

    showDialog(showTagDialog) {
        this.setState({
            showTagDialog,
        }, () => {
            if (!showTagDialog) {
                this.getTags();
            }
        });
    }

    getTags() {
        this.setState({
            listIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_ACCOUNT_TAGS_LIST,
                { queryParams: { object_id: this.props.organization.id } })
                .then(response => {
                    this.setState({
                        tagsList: response.records || [],
                        listIsLoading: false,
                        listError: null,
                    });
                })
                .catch(listError => {
                    this.setState({
                        listIsLoading: false,
                        listError,
                    });
                });
        });
    }

    componentDidMount() {
        this.getTags();
    }

    showJSON(showingTag) {
        this.setState({ showingTag, errorDetails: null });
    }

    showEditTag(editingTag) {
        this.setState({ editingTag });
    }

    showDeleteTag(deletingTag) {
        this.setState({ deletingTag });
    }

    deleteTag() {
        this.setState({
            deleteIsWorking: true,
        }, () => {
            this.request.exec(REQUESTS.REMOVE_ACCOUNT_TAG,
                { queryParams: { tag_id: this.state.deletingTag.tag_id } },
            ).then(() => {
                this.setState({
                    deleteIsWorking: false,
                    deleteError: null,
                    deletingTag: null,
                }, () => {
                    this.getTags();
                });
            }).catch(deleteError => {
                this.setState({
                    deleteIsWorking: false,
                    deleteError,
                });
            });
        });
    }

    showTagEventDetails(data) {
        this.setState({
            showingTag: {},
            detailsIsLoading: true,
            errorDetails: null,
        }, () => {
            this.request.exec(REQUESTS.GET_ACCOUNT_TAG_DETAILS,
                { queryParams: { event_id: data.event_id, tag_id: data.tag_id } })
                .then(response => {
                    this.setState({
                        showingTag: response,
                        detailsIsLoading: false,
                    });
                })
                .catch(errorDetails => {
                    this.setState({
                        errorDetails,
                        detailsIsLoading: false,
                    });
                });
        });
    }

    render() {
        const { onClose, organization } = this.props;
        const value = { ...this.state, ...{ object_id: organization.id } };

        return <Window title={`${organization?.details?.company}: аккаунт-теги`} onClose={onClose}>
            <AccountTagsModalContext.Provider value={value}>
                <div className={style.accountModal}>
                    <AccountTags showJSON={this.showJSON.bind(this)}
                                 showEditTag={this.showEditTag.bind(this)}
                                 showDeleteTag={this.showDeleteTag.bind(this)}
                                 addTag={this.showDialog.bind(this)}/>
                    <AccountTagsHistory object_id={organization.id}
                                        showTagDetails={this.showJSON.bind(this)}
                                        showTagEventDetails={this.showTagEventDetails.bind(this)}/>
                </div>
            </AccountTagsModalContext.Provider>
            {
                this.state.showTagDialog
                && <TagModal onClose={this.showDialog.bind(this, false)}
                             withoutPropositions
                             objectId={{ type: OBJECT_TYPES.ACCOUNT, id: organization.id }}/>
            }
            {
                this.state.editingTag
                && <TagModal onClose={this.showEditTag.bind(this, null)}
                             operation={Operations.EDIT}
                             initialData={this.state.editingTag}
                             objectId={{ type: OBJECT_TYPES.ACCOUNT, id: organization.id }}/>
            }
            {
                this.state.deletingTag
                && <Confirm isWorking={this.state.deleteIsWorking}
                            error={this.state.deleteError}
                            accept={this.deleteTag.bind(this)}
                            question={`Удалить тег ${this.state.deletingTag?.tag}?`}
                            onClose={this.showDeleteTag.bind(this, null)}/>
            }

            {
                this.state.showingTag
                && <JsonModal isLoading={this.state.detailsIsLoading}
                              error={this.state.errorDetails}
                              title={'Инфоромация о теге'}
                              obj={this.state.showingTag}
                              onClose={this.showJSON.bind(this, null)}/>
            }
        </Window>;
    }
}

interface IAccountTagsProps {
    addTag: () => void;
    showJSON: () => void;
    showEditTag: () => void;
    showDeleteTag: () => void;
}

const AccountTags = (props: IAccountTagsProps) => {
    const { addTag, showJSON, showEditTag, showDeleteTag } = props;
    const context = React.useContext(AccountTagsModalContext);

    return <div className={`${style.account_tags}`}>
        <h4>Теги: <Button onClick={addTag}>Добавить тег </Button></h4>
        <div>
            {
                context.listIsLoading && <Spin size={'s'}/>
            }
            {
                context.listError && <SimpleError error={context.listError}/>
            }
            {
                context.tagsList?.length && <table className={styleTable.table}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>tag</th>
                            <th>display_name</th>
                            <th>comment</th>
                            <th colSpan={3}/>
                        </tr>
                    </thead>
                    <tbody>
                        {
                            context.tagsList.map((element, index) => {
                                return <tr key={index}>
                                    <td>{index + 1}</td>
                                    <td>{element.tag}</td>
                                    <td>{element.display_name}</td>
                                    <td>{element.comment}</td>
                                    <td><Link onClick={showJSON.bind(null, element)}>info</Link></td>
                                    <td><Link onClick={showEditTag.bind(null, element)}>edit</Link></td>
                                    <td><Link onClick={showDeleteTag.bind(null, element)}>delete</Link></td>
                                </tr>;
                            })
                        }
                    </tbody>
                </table> || null
            }

        </div>
    </div>;
};

const AccountTagsHistory = (props) => {
    const context = React.useContext(AccountTagsModalContext);

    return <div className={`${style.account_tags_history}`}>
        <h4>История:</h4>
        <TagsHistory object_id={context.object_id}
                     historyType={HISTORY_TYPES.ACCOUNT}
                     isUpdating={false}
                     showTagDetails={props.showTagDetails}
                     showTagEventDetails={props.showTagEventDetails}/>
    </div>;
};
