import useStateMachine from '@cassiozen/usestatemachine';
import * as React from 'react';

import { UIStatusTypes } from '../../../../ui';
import { Button } from '../../../../ui/Button';
import { Link } from '../../../../ui/Link';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { AccountAddModal } from '../../../Settings/Wallets/AccountAddModal';
import { AccountDetailsModal } from '../../../Settings/Wallets/AccountDetailsModal';
import { IAccount } from '../../../Settings/Wallets/types';
import { SimpleError } from '../../../SimpleError';
import { REQUESTS } from '../../request';
import { DescriptionIcon } from '../utils/DescriptionIcon';
import styles from './index.css';

interface ICreateWalletsOrganizationProps {
    request: Request2;
}

export const CreateWalletsOrganization = (props: ICreateWalletsOrganizationProps) => {
    const { request } = props;
    const [state, send] = useStateMachine({
        selectStatus: {
            type: UIStatusTypes.info,
        },
    })({
        initial: 'initial',
        states: {
            initial: {
                on: {
                    INITIATE: 'loadingRootAccounts',
                },
            },
            inactive: {
                on: {
                    CREATE: 'creatingCompanyAccount',
                },
            },
            noRootAccountChosen: {
                on: {
                    CREATE: {
                        target: 'creatingCompanyAccount',
                        guard() {
                            return !!rootAccountName;
                        },
                    },
                },
                effect(send, update) {
                    update((context) => ({
                        ...context,
                        selectStatus: { type: UIStatusTypes.negative, text: 'Выберите очередь пожалуйста' },
                    }));

                    return () => {
                        update((context) => ({
                            ...context,
                            selectStatus: { type: UIStatusTypes.info },
                        }));
                    };
                },
            },
            loadingRootAccounts: {
                on: {
                    SUCCESS: 'inactive',
                    ERROR: 'loadRootAccountsError',
                },
                effect() {
                    setError(null);
                    (async () => { await fetchRootAccounts(); })();
                },
            },
            loadRootAccountsError: {
                on: {
                    RETRY: 'loadingRootAccounts',
                },
            },
            creatingCompanyAccount: {
                on: {
                    CREATED_AND_FETCHED: 'editCompanyAccount',
                    ERROR: 'showParentIdError',
                    NO_ROOT_ACCOUNT: 'noRootAccountChosen',
                },
                effect() {
                    if (!rootAccountName) {
                        send('NO_ROOT_ACCOUNT');
                    }

                    setError(null);
                    setAccount(null);
                },
            },
            showParentId: {
                on: {
                    CREATE: 'creatingCompanyAccount',
                },
            },
            showParentIdError: {
                on: {
                    CREATE: 'creatingCompanyAccount',
                },
            },
            editCompanyAccount: {
                on: {
                    CLOSE: 'showParentId',
                },
            },
        },
    });

    const [account, setAccount] = React.useState<IAccount | null>(null);
    const [error, setError] = React.useState<Error | null>(null);
    const [rootAccountName, setRootAccountName] = React.useState<string | null>(null);
    const [newAccountId, setNewAccountId] = React.useState<string | null>(null);
    const [selectOptions, setSelectOptions] = React.useState<IOptionInfo[]>([]);

    const fetchNewAccountWithRetry = () => {
        const RETRY_LIMIT = 3;
        let retryCounter = 0;

        return async (newAccountId) => {
            setNewAccountId(newAccountId);
            try {
                if (rootAccountName) {
                    const response = await request.exec(REQUESTS.GET_ACCOUNTS, {
                        queryParams: {
                            account_id: encodeURIComponent(newAccountId),
                        },
                    });

                    if (response.hasOwnProperty('id')) {
                        setAccount(response);
                        send('CREATED_AND_FETCHED');
                    } else {
                        if (retryCounter < RETRY_LIMIT) {
                            retryCounter += 1;
                            await fetchNewAccount(newAccountId);
                        } else {
                            send('ERROR');
                        }
                    }
                }
            } catch (error) {
                setError(error);
                send('ERROR');
            }
        };
    };

    const fetchNewAccount = fetchNewAccountWithRetry();

    const fetchRootAccounts = async () => {
        try {
            const response = await request.exec(REQUESTS.GET_WALLETS);

            const selectOptions = response.accounts.reduce((acc, wallet) => {
                if (!wallet?.meta?.parent_id) {
                    acc.push({ text: wallet?.meta?.hr_name ?? wallet.name, value: wallet.name });
                }

                return acc;
            }, []);

            setSelectOptions(selectOptions);
            send('SUCCESS');
        } catch (e) {
            send('ERROR');
            setError(e);
        }
    };

    return <>
        {error
            ? <SimpleError error={error}/>
            : null
        }
        {state.value === 'editCompanyAccount' && account
            ? <AccountDetailsModal onClose={() => {
                setAccount(null);
                send('CLOSE');
            }}
                                   account={account}/>
            : null
        }
        {state.value === 'creatingCompanyAccount' && rootAccountName
            ? <AccountAddModal accountName={rootAccountName}
                               onClose={async (accountId) => {
                                   await fetchNewAccount(accountId);
                               }}/>
            : null
        }
        <div className={styles.createOrganizationContainer}>
            {selectOptions?.length
                ? <Select status={state.context.selectStatus}
                          options={selectOptions}
                          placeholder={'Выберите очередь для кошелька компании'}
                          onSelect={(name) => {
                              setRootAccountName(name as unknown as string);
                          }}/>
                : <>
                    <Button isLoading={state.value === 'loadingRootAccounts'}
                            onClick={() => {
                                if (state.value === 'loadRootAccountsError') {
                                    send('RETRY');
                                } else {
                                    send('INITIATE');
                                }
                            }}>
                        Загрузить очереди
                    </Button>
                </>
            }
            <DescriptionIcon className={styles.organizationHint}>
                <p><b>Кошелек компании</b> - второй уровень кошельков</p>
                <img src={'https://carsharing.s3.yandex.net/admin/static/b2b/wallet-organization-hint.png'}
                     alt={'Скриншот кошелька компании в старых кошельках'}/>
                <p>
                    <b>Очередь</b> - верхний уровень кошельков
                </p>
                <img src={'https://carsharing.s3.yandex.net/admin/static/b2b/wallet-queue-hint.png'}
                     alt={'Скриншот очереди в старых кошельках'}/>
            </DescriptionIcon>
            <Button disabled={state.value === 'initial'}
                    onClick={() => {
                        send('CREATE');
                    }}>
                Создать кошелек компании
            </Button>
            {state.value === 'showParentId' || state.value === 'showParentIdError'
                ? <>
                    {state.value === 'showParentIdError'
                        ? <span className={`${styles.linkLabel} ${styles.editAccountError}`}>
                            Не удалось открыть форму редактирования кошелька
                        </span>
                        : null
                    }
                    <Link href={`#/b2b/search-wallets?organization_id=${newAccountId}`}
                          className={styles.linkLabel}>
                        Ссылка на созданный кошелек
                    </Link>
                </>
                : null
            }
        </div>
    </>;
};
