import * as React from 'react';
import VirtualList, { ItemStyle } from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { useVirtualListDynamicHeight } from '../../../../hooks/useVirtualListDynamicHeight';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { rubs } from '../../../../utils/rubs';
import Spin from '../../../Spin';
import { REQUESTS } from '../../request';
import * as mainStyles from '../index.css';
import { IBalanceHistoryData, VIRTUALLIST_TABLE_HEIGHT } from '../types';
import * as styles from './index.css';

interface IBalanceHistoryProps {
    request: Request2;
    onClose: () => void;
    accountId: string;
}

export const BalanceHistory = (props: IBalanceHistoryProps) => {
    const { request, onClose, accountId } = props;

    const responseHandler = (_response) => {
        return _response?.operations_history?.map((operation) => {
            return {
                company: operation?.account_meta?.company ?? EMPTY_DATA,
                limit: operation?.account_meta?.soft_limit ? rubs(operation?.account_meta?.soft_limit) : EMPTY_DATA,
                timestamp: operation?.timestamp * ONE_SECOND ?? 0,
                limitDiff: operation?.account_meta?.soft_limit_diff
                    ? rubs(operation?.account_meta?.soft_limit_diff)
                    : EMPTY_DATA,
            };
        }) as IBalanceHistoryData[];
    };

    const requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_BALANCE,
            requestOptions: {
                queryParams: {
                    account_id: accountId,
                },
            },
        };
    }, [accountId]);

    const [
        isLoading,
        response,
        error,
        getBalanceHistory,
    ] = useRequestHandler<IBalanceHistoryData[]>(request, requestOptions, responseHandler);

    React.useEffect(() => {
        getBalanceHistory();
    }, []);

    return <Window className={styles.balanceHistory}
                   title={'История пополнения кошелька'}
                   error={error}
                   onClose={onClose.bind(null)}>
        {isLoading
            ? <Spin/>
            : !error && response
                ? <BalanceHistoryTable data={response}/>
                : null
        }
    </Window>;
};

interface IBalanceHistoryTable {
    data: IBalanceHistoryData[];
}

const ITEM_SIZE = 60;
const ITEM_LIMIT = 10;
const ITEM_HEIGHT_PADDING = 5;

const BalanceHistoryTable = React.memo((props: IBalanceHistoryTable) => {
    const { data } = props;
    const [rowHeights, setHeightFromRef] = useVirtualListDynamicHeight({
        defaultItemSize: ITEM_SIZE,
        initialState: new Array(data?.length).fill(ITEM_SIZE),
    });

    const tableHeight = rowHeights.slice(0, ITEM_LIMIT).reduce((a, b) => {
        a += b;

        return a;
    }, 0) + ITEM_HEIGHT_PADDING ?? VIRTUALLIST_TABLE_HEIGHT;

    return <div>
        <div className={`${mainStyles.header} ${styles.balanceHistoryRow}`}>
            <span>Дата</span>
            <span>Компания</span>
            <span>Общая сумма поступлений</span>
            <span>Изменение</span>
        </div>
        {data?.length
            ? <VirtualList width={'100%'}
                           height={tableHeight}
                           itemCount={data?.length}
                           itemSize={(index) => {
                               return rowHeights[index];
                           }}
                           renderItem={({ index, style }) => {
                               const operation = data[index];

                               return <BalanceHistoryTableItem setHeightFromRef={setHeightFromRef.bind(null, index)}
                                                               operation={operation}
                                                               style={style}
                                                               key={index}/>;
                           }}/>
            : <NoInformation/>
        }
    </div>;
});

interface IBalanceHistoryTableItem {
    operation: IBalanceHistoryData;
    style: ItemStyle;
    setHeightFromRef: (ref) => void;
}

const BalanceHistoryTableItem = (props: IBalanceHistoryTableItem) => {
    const { operation, style, setHeightFromRef } = props;
    const companyRef = React.useRef<HTMLDivElement>(null);

    React.useEffect(() => {
        setHeightFromRef(companyRef);
    }, []);

    return <div style={style}
                className={`${mainStyles.row} ${styles.balanceHistoryRow}`}>
        <span>
            <FormatDate value={operation?.timestamp}/>
        </span>
        <span ref={companyRef}>
            {operation?.company}
        </span>
        <span>
            {operation?.limit}
        </span>
        <span>
            {operation?.limitDiff}
        </span>
    </div>;
};
