import React from 'react';
import VirtualList, { ItemStyle } from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { rubs } from '../../../../utils/rubs';
import Spin from '../../../Spin';
import { REQUESTS } from '../../request';
import * as mainStyles from '../index.css';
import { IOverdraftHistoryData, VIRTUALLIST_TABLE_HEIGHT } from '../types';
import * as styles from './index.css';

interface IOverdraftHistoryProps {
    request: Request2;
    onClose: () => void;
    accountId: string;
}

export const OverdraftHistory = (props: IOverdraftHistoryProps) => {
    const { request, onClose, accountId } = props;

    const responseHandler = (_response) => {
        return _response?.operations_history?.reduce((prev, operation, i) => {
            const overdraft = operation?.account_meta?.balance_info?.overdraft;
            const prevOverdraft = _response?.operations_history[i + 1]?.account_meta?.balance_info?.overdraft;
            const overdraftDiff = prevOverdraft >= 0 && overdraft >= 0
                ? overdraft - prevOverdraft ? rubs(overdraft - prevOverdraft) : null
                : overdraft ? rubs(overdraft) : null;

            if (overdraftDiff) {
                prev.push({
                    company: operation?.account_meta?.company ?? EMPTY_DATA,
                    overdraft: overdraft ? rubs(overdraft) : EMPTY_DATA,
                    timestamp: operation?.timestamp * ONE_SECOND ?? 0,
                    overdraftDiff,
                });
            }

            return prev;
        }, []) as IOverdraftHistoryData[];
    };

    const requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_BALANCE,
            requestOptions: {
                queryParams: {
                    account_id: accountId,
                },
            },
        };
    }, [accountId]);

    const [
        isLoading,
        response,
        error,
        getBalanceHistory,
    ] = useRequestHandler<IOverdraftHistoryData[]>(request, requestOptions, responseHandler);

    React.useEffect(() => {
        getBalanceHistory();
    }, []);

    return <Window className={styles.overdraftHistory}
                   title={'История изменений овердрафта'}
                   error={error}
                   onClose={onClose.bind(null)}>
        {isLoading
            ? <Spin/>
            : !error && response
                ? <OverdraftHistoryTable data={response}/>
                : null
        }
    </Window>;
};

interface IOverdraftHistoryTable {
    data: IOverdraftHistoryData[];
}

const ITEM_SIZE = 60;
const ITEM_LIMIT = 10;
const ITEM_HEIGHT_PADDING = 5;

const OverdraftHistoryTable = React.memo((props: IOverdraftHistoryTable) => {
    const { data } = props;
    const tableHeight = new Array(data?.length).fill(ITEM_SIZE).slice(0, ITEM_LIMIT).reduce((a, b) => {
        a += b;

        return a;
    }, 0) + ITEM_HEIGHT_PADDING ?? VIRTUALLIST_TABLE_HEIGHT;

    return <div>
        <div className={`${mainStyles.header} ${styles.overdraftHistoryRow}`}>
            <span>Дата</span>
            <span>Компания</span>
            <span>Текущий овердрафт</span>
            <span>Изменение</span>
        </div>
        {data?.length
            ? <VirtualList width={'100%'}
                           height={tableHeight}
                           itemCount={data?.length}
                           itemSize={ITEM_SIZE}
                           renderItem={({ index, style }) => {
                               const operation = data[index];

                               return <OverdraftHistoryTableItem operation={operation}
                                                                 style={style}
                                                                 key={index}/>;
                           }}/>
            : <NoInformation/>
        }
    </div>;
});

interface IOverdraftHistoryTableItem {
    operation: IOverdraftHistoryData;
    style: ItemStyle;
}

const OverdraftHistoryTableItem = (props: IOverdraftHistoryTableItem) => {
    const { operation, style } = props;

    return <div style={style}
                className={`${mainStyles.row} ${styles.overdraftHistoryRow}`}>
        <span>
            <FormatDate value={operation?.timestamp}/>
        </span>
        <span>
            {operation?.company}
        </span>
        <span>
            {operation?.overdraft}
        </span>
        <span>
            {operation?.overdraftDiff}
        </span>
    </div>;
};
