import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { ONE_SECOND } from '../../../../constants';
import { useRequestQueueHandler } from '../../../../hooks/useRequestQueueHandler';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../../../models/session';
import { DownloadSessionsButton } from '../../../../ui/DownloadSessionsButton';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { SimpleError } from '../../../SimpleError';
import { REQUESTS } from '../../request';
import { SinceUntilSearch } from '../../utils/SinceUntilSearch';
import { SinceUntilSearchInitialSince, SinceUntilSearchInitialUntil } from '../../utils/SinceUntilSearch/constants';
import * as mainStyles from '../index.css';
import { VIRTUALLIST_TABLE_HEIGHT } from '../types';
import * as styles from './index.css';

interface IRidesHistory {
    request: Request2;
    userIds: string[];
    onClose: () => void;
    accountName: string;
    BlockRules: any;
}

interface IB2BRide {
    userPrintName: string;
    userId: string;
    start: number;
    finish: number;
    offerName: string;
    totalPrice: number;
    sessionId: string;
}

const ITEM_SIZE = 70;

const mergeSessions = (allUsersSessions, accountName): SessionsHistoryItem => {
    if (allUsersSessions) {
        const result = deepCopy(allUsersSessions[0]);
        result.sessions = [];

        for (let i = 0; i < allUsersSessions.length; i++) {
            const userB2bSessions = allUsersSessions[i]?.sessions?.filter((session, index) => {
                const selectedCharge = SessionHistoryInfoHandler.getSelectedCharge.call(allUsersSessions[i], index);

                return selectedCharge === accountName;
            }) ?? [];

            result.sessions = result.sessions.concat(userB2bSessions);
        }

        result.sessions?.sort((a, b) => {
            const AStart = SessionHistoryInfoHandler.getStart.call(result, result.sessions.indexOf(a));
            const BStart = SessionHistoryInfoHandler.getStart.call(result, result.sessions.indexOf(b));

            return BStart - AStart;
        });

        return result;
    }

    return {} as SessionsHistoryItem;

};

const getModalContent = (response, accountName): IB2BRide[] => {
    return response?.reduce((acc: IB2BRide[], res) => {
        res?.sessions?.forEach((session, index) => {
            const selectedCharge = SessionHistoryInfoHandler.getSelectedCharge.call(res, index);

            if (selectedCharge === accountName) {
                acc.push({
                    userPrintName: SessionHistoryInfoHandler.getUserPrintName.call(res, index),
                    userId: SessionHistoryInfoHandler.getUserId.call(res, index),
                    start: SessionHistoryInfoHandler.getStart.call(res, index),
                    finish: SessionHistoryInfoHandler.getFinish.call(res, index),
                    offerName: SessionHistoryInfoHandler.getOfferName.call(res, index),
                    totalPrice: SessionHistoryInfoHandler.getTotalPrice.call(res, index),
                    sessionId: SessionHistoryInfoHandler.getSessionId.call(res, index),
                });
            }
        });

        return acc;
    }, [])?.sort((a, b) => b.start - a.start) ?? [];
};

export const RidesHistory = (props: IRidesHistory) => {
    const { request, userIds, onClose, accountName, BlockRules } = props;

    const getQueue = (since, until) => {
        return userIds.map((user_id) => {
            return {
                requestName: REQUESTS.GET_B2B_RIDES,
                requestOptions: {
                    queryParams: {
                        user_id,
                        since: Math.round(since / ONE_SECOND),
                        until: Math.round(until / ONE_SECOND),
                    },
                },
            };
        });
    };

    const [currentQueue, setCurrentQueue] = React.useState(
        getQueue(SinceUntilSearchInitialSince, SinceUntilSearchInitialUntil),
    );

    const [
        isLoading,
        response,
        errors,
        getSessionHistory,
    ] = useRequestQueueHandler<SessionsHistoryItem[]>(request, currentQueue);
    const modalContent = getModalContent(response, accountName);
    const sessionDownload = mergeSessions(response, accountName);

    React.useEffect(() => {
        getSessionHistory();
    }, [currentQueue]);

    const isSessionDownloadAvailable = !!Object.keys(sessionDownload)?.length;

    return <Window className={styles.ridesHistory}
                   onClose={onClose.bind(null)}
                   title={'История корпоративных поездок'}>
        {errors.map((error, index) => {
            return <SimpleError key={index} error={error}/>;
        })}

        <div className={styles.dates}>
            <SinceUntilSearch isSearching={isLoading}
                              onSearch={({ since, until }) => {
                                  setCurrentQueue(getQueue(since, until));
                              }}/>
            {isSessionDownloadAvailable
                && BlockRules?.WalletSessions
                && <DownloadSessionsButton sessionHistory={sessionDownload}
                                           className={styles.downloadSessions}
                                           spreadsheetTitle={`sessions-${accountName}`}/>}
        </div>

        {!errors?.length && response && response?.length
            ? <div className={styles.table}>
                <div className={`${mainStyles.header} ${styles.ridesHistoryRow}`}>
                    <span>ФИО</span>
                    <span>Дата начала поездки</span>
                    <span>Дата окончания поездки</span>
                    <span>Тариф</span>
                    <span className={styles.money}>
                        Стоимость
                    </span>
                    <span>Перейти к сессии</span>
                </div>
                <VirtualList width={'100%'}
                             height={VIRTUALLIST_TABLE_HEIGHT}
                             itemCount={modalContent?.length}
                             itemSize={ITEM_SIZE}
                             renderItem={({ index, style }) => {
                                 const session: IB2BRide = modalContent?.[index] ?? {} as IB2BRide;

                                 return <div key={index}
                                             style={style}
                                             className={`${mainStyles.row} ${styles.ridesHistoryRow}`}>
                                     <span>
                                         <Link href={`#/clients/${session.userId}/info`}
                                               target={'_blank'}>
                                             {session.userPrintName}
                                         </Link>
                                     </span>
                                     <span>
                                         <FormatDate withSecond value={session.start}/>
                                     </span>
                                     <span>
                                         <FormatDate withSecond value={session.finish}/>
                                     </span>
                                     <span>
                                         {session.offerName}
                                     </span>
                                     <span className={styles.money}>
                                         {session.totalPrice}
                                     </span>
                                     <span>
                                         <Link href={`#/session/${session.sessionId}/billing`}
                                               target={'_blank'}>
                                    подробнее
                                         </Link>
                                     </span>
                                 </div>;
                             }}/>
            </div>
            : <NoInformation/>
        }
    </Window>;
};
