import * as React from 'react';
import XLSX from 'xlsx';

import { Dict } from '../../../../../../types';
import { EMPTY_DATA } from '../../../../../constants';
import { BasicUserInfo, UserInfoHandler } from '../../../../../models/user';
import { Input } from '../../../../../ui/Input';
import { Link } from '../../../../../ui/Link';
import { NoInformation } from '../../../../../ui/NoInformation';
import { PayRoadIcon } from '../../../utils/PayRoadIcon/component';
import CreateWalletFromResults from '../../CreateWalletFromResults';
import { IWalletAccounts, IWalletLimit } from '../../types';
import { TShowRideHistoryForIds } from '../component';
import styles from './index.css';

interface ISearchResultsWalletLimit {
    walletLimit?: IWalletLimit;
    walletAccounts: Dict<IWalletAccounts[]>;
    showRideHistoryForIds: TShowRideHistoryForIds;
    showWalletBalanceHistory: (accountId: number) => void;
    showWalletsHistory: (accountId: number) => void;
    BlockRules?: Dict<boolean>;
    children: React.ReactElement;
    setWalletNameForEdit: (walletName: string) => void;
    setIdToCreateAccount: (walletName: string) => void;
    setFilteredWalletAccounts: (walletAccounts: Dict<IWalletAccounts[]>) => void;
}

const downloadUserXLSX = (data: { limit: string; users: BasicUserInfo[] }) => {
    const wb = XLSX.utils.book_new();
    const ws_data: any = [];
    ws_data.push(
        [
            'id',
            'last_name',
            'first_name',
            'paternal_name',
            'phone',
            'username',
            'status',
        ],
        ...data?.users?.map((user) => {
            return [
                UserInfoHandler.getId.call(user) ?? EMPTY_DATA,
                UserInfoHandler.getLastName.call(user) ?? EMPTY_DATA,
                UserInfoHandler.getName.call(user) ?? EMPTY_DATA,
                UserInfoHandler.getPn.call(user) ?? EMPTY_DATA,
                UserInfoHandler.getPhone.call(user),
                UserInfoHandler.getUsername.call(user),
                UserInfoHandler.getStatus.call(user) ?? EMPTY_DATA,
            ];
        }));

    const ws = XLSX.utils.aoa_to_sheet(ws_data);
    const MAX_SHEET_NAME_LENGTH = 31;
    XLSX.utils.book_append_sheet(wb, ws, data.limit?.substr(0, MAX_SHEET_NAME_LENGTH) ?? 'sheet');

    XLSX.writeFile(wb, `users-${data.limit}.xlsx`);
};

export const SearchResultsWalletLimit = (props: ISearchResultsWalletLimit) => {
    const [isExpandedAccounts, expandAccounts] = React.useState(false);
    const [filterValue, setFilterValue] = React.useState('');
    const name = props.walletLimit?.name ?? '';
    const meta = props.walletLimit?.meta;
    const organizationCreatorArray = props.walletLimit?.organization_creator;

    function containsUser(user, filter) {
        const isRegistrationRequest = !UserInfoHandler.getId.call(user)
            && UserInfoHandler.getPhone.call(user);

        return !filter || (isRegistrationRequest
            ? isRegistrationRequest.includes(filter)
            : user?.info?.toLowerCase().includes(filter.toLowerCase()));
    }

    function getFilteredLimits(filter) {
        const { walletLimit, walletAccounts } = props;
        const accounts = walletLimit && walletAccounts[walletLimit?.name] || [];

        return accounts.filter((account) => containsUser(account.users[0], filter));
    }

    function onFilterChange(value) {
        const { setFilteredWalletAccounts, walletLimit, walletAccounts } = props;
        setFilterValue(value);
        if (walletLimit) {
            setFilteredWalletAccounts({
                ...walletAccounts,
                [walletLimit.name]: getFilteredLimits(value),
            });
        }
    }

    React.useEffect(() => {
        expandAccounts(false);
    }, [name]);

    const {
        walletAccounts,
        showRideHistoryForIds,
        showWalletBalanceHistory,
        showWalletsHistory,
        BlockRules,
        setWalletNameForEdit,
        setIdToCreateAccount,
        walletLimit,
    } = props;
    //registration requests don't return id for user, only phones

    const { users, userIds } = React.useMemo(() => {
        const users: BasicUserInfo[] = [];
        const userIds: string[] = [];

        walletAccounts[name]?.forEach((walletAccount) => {
            walletAccount.users?.forEach((user) => {
                const userId = UserInfoHandler.getId.call(user);
                if (userId) {
                    userIds.push(userId);
                }

                users.push(user);
            });
        });

        return { users, userIds };
    }, [name]);

    const childrenProps = {
        walletName: name,
    };
    const children = React.cloneElement(props.children, childrenProps);

    return <div className={styles.walletLimit}>
        <div className={styles.walletLimitHeader}>
            <span className={styles.leftSide}>
                <span>
                    <span>{name}</span>
                    <span className={styles.title}>{meta?.parent_id ? ` — ${meta?.parent_id}` : ''}</span>
                </span>
                {meta?.hr_name && <span className={styles.title}>{meta.hr_name}</span>}
            </span>
            <span className={`${styles.walletLimitAdditionals} ${styles.walletLimitInfo}`}>
                <span className={styles.filter}>
                    <span className={styles.title}>Фильтр: </span>
                    <span>{meta?.offers_filter ?? EMPTY_DATA}</span>
                </span>
                <span>
                    <span className={styles.title}>Владелец: </span>
                    <span>
                        {organizationCreatorArray?.length
                            ? organizationCreatorArray?.map((organizationCreator, index) => (
                                <>
                                    <Link target={'_blank'}
                                          key={index}
                                          href={`#/clients/${UserInfoHandler.getId.call(organizationCreator)}`}>
                                        {UserInfoHandler.getUsername.call(organizationCreator)}
                                    </Link>
                                    <br/>
                                </>
                            ))
                            : EMPTY_DATA
                        }
                    </span>
                </span>
                <span>
                    {/*>Юля: в дескрипшене опциональное поле enable_toll_road, если его нет, то считаем, что по умолчанию дороги включены*/}
                    Оплата дорог: <PayRoadIcon enable={meta?.enable_toll_roads !== undefined
                        ? meta?.enable_toll_roads
                        : true}/>
                </span>
            </span>
            <span className={`${styles.walletLimitAdditionals} ${styles.walletLimitActions}`}>
                <Link onClick={showWalletBalanceHistory.bind(null, meta?.parent_id)}>
                    История баланса (организация)
                </Link>
                <Link onClick={showWalletsHistory.bind(null, walletLimit?.id)}>
                    История изменения тарифов
                </Link>
                {userIds?.length
                    ? <Link onClick={showRideHistoryForIds.bind(null, userIds, name)}>
                        История поездок
                    </Link>
                    : null
                }
                {BlockRules?.WalletUsers && <Link onClick={downloadUserXLSX.bind(null, { limit: name, users })}>
                    Скачать список пользователей
                </Link>}
                {BlockRules?.EditWalletB2B && <Link onClick={() => {
                    setWalletNameForEdit(name);
                }}>
                    Редактировать
                </Link>}
            </span>
            <span className={styles.expandLink}
                  onClick={expandAccounts.bind(null, !isExpandedAccounts)}>
                {isExpandedAccounts ? 'Свернуть' : 'Развернуть'}
            </span>
        </div>
        {isExpandedAccounts
            && <div className={styles.controls}>
                <Input className={styles.authorFilter}
                       onChange={onFilterChange}
                       value={filterValue}
                       placeholder={'Введите пользователя'}/>
                <CreateWalletFromResults parentWalletId={name}
                                         addNewWallet={setIdToCreateAccount}
                                         title={'Добавить кошелек'}/>
            </div>}
        {walletAccounts ?
            isExpandedAccounts
                ? children
                : null
            : <NoInformation/>
        }
    </div>;
};
