import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { IAccountHistoryItem } from '../../../Settings/Wallets/types';
import { maskId } from '../../../Settings/Wallets/utils';
import Spin from '../../../Spin';
import { REQUESTS } from '../../request';
import * as mainStyles from '../index.css';
import { IUserLinksData, VIRTUALLIST_TABLE_HEIGHT } from '../types';
import * as styles from './index.css';

interface IUserLinksHistoryProps {
    request: Request2;
    onClose: () => void;
    accountId: string;
}

export const UserLinksHistory = (props: IUserLinksHistoryProps) => {
    const { request, onClose, accountId } = props;

    const responseHandler = (_response: IAccountHistoryItem) => {
        return _response?.links_history?.map((link) => {
            return {
                action: link?.history_action ?? EMPTY_DATA,
                userId: link?.user_id ?? EMPTY_DATA,
                timestamp: link?.history_timestamp * ONE_SECOND ?? 0,
                authorUserId: link?.history_user_id ?? EMPTY_DATA,
            };
        })?.reverse();
    };

    const options = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_USER_LINKS,
            requestOptions: {
                queryParams: {
                    account_id: accountId,
                },
            },
        };
    }, [accountId]);
    const [
        isLoading,
        response,
        error,
        getUserLinkHistory,
    ] = useRequestHandler<IUserLinksData[]>(request, options, responseHandler);

    React.useEffect(() => {
        getUserLinkHistory();
    }, []);

    return <Window error={error}
                   onClose={onClose.bind(null)}
                   title={'История привязки пользователей'}>
        {isLoading
            ? <Spin/>
            : !error && response
                ? <UserLinksHistoryTable data={response}/>
                : null
        }
    </Window>;
};

interface IUserLinksHistoryTableProps {
    data: IUserLinksData[];
}

const ITEM_SIZE = 50;

const UserLinksHistoryTable = React.memo((props: IUserLinksHistoryTableProps) => {
    const { data } = props;

    return <div>
        <div className={`${mainStyles.header} ${styles.userLinksHistoryRow}`}>
            <span>Дата</span>
            <span>Действие</span>
            <span>Автор</span>
            <span>Пользователь</span>
        </div>
        {data?.length
            ? <VirtualList width={'100%'}
                           height={VIRTUALLIST_TABLE_HEIGHT}
                           itemCount={data?.length}
                           itemSize={ITEM_SIZE}
                           renderItem={({ index, style }) => {
                               const userLink = data[index];

                               return <div key={index}
                                           style={style}
                                           className={`${mainStyles.row} ${styles.userLinksHistoryRow}`}>
                                   <span>
                                       <FormatDate value={userLink?.timestamp}
                                                   withSecond/>
                                   </span>
                                   <span>
                                       {userLink?.action}
                                   </span>
                                   <span>
                                       <Link href={`#/clients/${userLink?.authorUserId}/info`}
                                             target={'_blank'}>
                                           {maskId(userLink?.authorUserId)}
                                       </Link>
                                   </span>
                                   <span>
                                       <Link href={`#/clients/${userLink?.userId}/info`}
                                             target={'_blank'}>
                                           {maskId(userLink?.userId)}
                                       </Link>
                                   </span>
                               </div>;
                           }}/>
            : <NoInformation/>
        }
    </div>;
});
