import * as React from 'react';
import VirtualList, { ItemStyle } from 'react-tiny-virtual-list';

import { ONE_SECOND } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { useVirtualListDynamicHeight } from '../../../../hooks/useVirtualListDynamicHeight';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { maskId } from '../../../Settings/Wallets/utils';
import Spin from '../../../Spin';
import { REQUESTS } from '../../request';
import * as mainStyles from '../index.css';
import { VIRTUALLIST_TABLE_HEIGHT } from '../types';
import * as styles from './index.css';

interface IWalletsHistoryProps {
    request: Request2;
    onClose: () => void;
    accountId: string;
}

export const WalletsHistory = (props: IWalletsHistoryProps) => {
    const { request, onClose, accountId } = props;

    const responseHandler = (_response) => {

        return _response?.operations_history?.reduce((prev, operation, i) => {
            const offersFilter = operation?.meta?.offers_filter;
            const prevOffersFilter = _response?.operations_history[i + 1]?.meta?.offers_filter;

            if (offersFilter !== prevOffersFilter) {
                prev.push({
                    timestamp: operation?.timestamp * ONE_SECOND ?? 0,
                    user_id: operation?.user_id,
                    action: operation?.action,
                    offersFilter,
                });
            }

            return prev;
        }, []);
    };

    const requestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_WALLETS_HISTORY,
            requestOptions: {
                queryParams: {
                    id: accountId,
                },
            },
        };
    }, [accountId]);

    const [
        isLoading,
        response,
        error,
        getWalletsHistory,
    ] = useRequestHandler<Record<string, any>[]>(request, requestOptions, responseHandler);

    React.useEffect(() => {
        getWalletsHistory();
    }, []);

    return <Window className={styles.walletsHistory}
                   title={'История изменения тарифов'}
                   error={error}
                   onClose={onClose.bind(null)}>
        {isLoading
            ? <Spin/>
            : !error && response
                ? <WalletsHistoryTable data={response}/>
                : null
        }
    </Window>;
};

interface IWalletsHistoryTable {
    data: any[];
}

const ITEM_SIZE = 60;
const ITEM_LIMIT = 10;
const ITEM_HEIGHT_PADDING = 5;

const WalletsHistoryTable = React.memo((props: IWalletsHistoryTable) => {
    const { data } = props;
    const [rowHeights, setHeightFromRef] = useVirtualListDynamicHeight({
        defaultItemSize: ITEM_SIZE,
        initialState: new Array(data?.length).fill(ITEM_SIZE),
    });

    const tableHeight = rowHeights.slice(0, ITEM_LIMIT).reduce((a, b) => {
        a += b;

        return a;
    }, 0) + ITEM_HEIGHT_PADDING ?? VIRTUALLIST_TABLE_HEIGHT;

    return <div>
        <div className={`${mainStyles.header} ${styles.walletsHistoryRow}`}>
            <span>Дата</span>
            <span>Изменил</span>
            <span>Текущий тариф</span>
        </div>
        {data?.length
            ? <VirtualList width={'100%'}
                           height={tableHeight}
                           itemCount={data?.length}
                           itemSize={(index) => {
                               return rowHeights[index];
                           }}
                           renderItem={({ index, style }) => {
                               const operation = data[index];

                               return <WalletsHistoryTableItem setHeightFromRef={setHeightFromRef.bind(null, index)}
                                                               operation={operation}
                                                               style={style}
                                                               key={index}/>;
                           }}/>
            : <NoInformation/>
        }
    </div>;
});

interface IWalletsHistoryTableItem {
    operation: any;
    style: ItemStyle;
    setHeightFromRef: (ref) => void;
}

const WalletsHistoryTableItem = (props: IWalletsHistoryTableItem) => {
    const { operation, style, setHeightFromRef } = props;
    const offersFilterRef = React.useRef<HTMLDivElement>(null);

    React.useEffect(() => {
        setHeightFromRef(offersFilterRef);
    }, []);

    return <div style={style}
                className={`${mainStyles.row} ${styles.walletsHistoryRow}`}>
        <span>
            <FormatDate value={operation?.timestamp}/>
        </span>
        <span>
            <Link href={`#/clients/${operation?.user_id}/info`}
                  target="_blank"
                  title={operation?.user_id}>
                {maskId(operation?.user_id)}
            </Link>
        </span>
        <span ref={offersFilterRef}>
            {operation?.offersFilter}
        </span>
    </div>;
};
