import * as React from 'react';

import { Button } from '../../../ui/Button';
import { NoInformation } from '../../../ui/NoInformation';
import Select from '../../../ui/Select';
import { isLocalhost } from '../../../utils/frontendEnvironment';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import { B2B_REQUESTS, REQUESTS } from '../request';
import ByNameSearchResults from './ByNameSearchResults';
import styles from './index.css';
import WalletsSearchResults from './SearchResults';
import { ISearchWalletsResults, SEARCH_ID_TYPE, SEARCH_NAME_TYPE, TSearchLinkArray } from './types';
import WalletsSearchBar from './WalletSearchBar';

interface ISearchWalletsProps {
    location: {
        search: string;
    };
}

interface ISearchWalletsState {
    error: Error | null;
    isSearching: boolean;
    searchQuery: string;
    searchByIdResults: any;
    searchByNameResults: TSearchLinkArray | null;
    searchIdType: SEARCH_ID_TYPE | null;
    searchNameType: SEARCH_NAME_TYPE | null;
    account_name: string;
}

const organizationCreatorTag = 'organization_creator';

export class SearchWallets extends React.Component<ISearchWalletsProps, ISearchWalletsState> {
    state: ISearchWalletsState = {
        error: null,
        isSearching: false,
        searchQuery: '',
        searchByIdResults: null,
        searchByNameResults: null,
        searchIdType: null,
        searchNameType: null,
        account_name: 'b2b_drive',
    };

    request: Request2;

    constructor(props) {
        super(props);
        this.request = new Request2({ requestConfigs: B2B_REQUESTS });
    }

    onChangeSearch(searchIdType: SEARCH_ID_TYPE, searchQuery: string) {
        this.setState({
            searchQuery,
            searchIdType,
            searchNameType: null,
        });
    }

    onChangeSearchByName(searchNameType: SEARCH_NAME_TYPE, searchQuery: string) {
        this.setState({
            searchQuery,
            searchIdType: null,
            searchNameType,
        });
    }

    setError(error) {
        this.setState({
            error,
        });
    }

    async searchButton() {
        if (this.state.searchIdType) {
            location.href = `#/b2b/search-wallets?${this.state.searchIdType}=${this.state.searchQuery}`;
        }

        if (this.state.searchNameType) {
            location.hash = '#/b2b/search-wallets';

            this.setState({
                error: null,
                isSearching: true,
                searchByIdResults: null,
            });
            const searchResult = await this.onSearch(
                this.state.searchQuery,
                this.state.searchNameType,
                this.request,
                this.setError.bind(this),
            );
            this.setState({
                searchByNameResults: searchResult,
                isSearching: false,
            });
        }
    }

    async searchById() {
        this.setState({
            searchByNameResults: null,
            error: null,
            isSearching: true,
        });
        if (this.state.searchIdType) {
            const searchResult = await this.onSearch(
                this.state.searchQuery,
                this.state.searchIdType,
                this.request,
                this.setError.bind(this),
            );
            this.setState({
                searchByIdResults: searchResult,
                isSearching: false,
            });
        }
    }

    async onSearch(
        searchQuery: string,
        type: SEARCH_ID_TYPE | SEARCH_NAME_TYPE,
        request: Request2,
        errorHandler: (error: Error) => void,
    ) {
        switch (type) {
        case SEARCH_ID_TYPE.WALLET_LIMIT_ID:
            return await this.searchWalletLimit(searchQuery, request, errorHandler);
        case SEARCH_ID_TYPE.ORGANIZATION_ID:
            return await this.searchOrganizationsById(searchQuery, request, errorHandler);
        case SEARCH_NAME_TYPE.ORGANIZATION_NAME:
            return await this.searchOrganizationByName(searchQuery, request, errorHandler);
        default:
            return null;
        }
    }

    static matchesSearchQuery = (companyNameField: string, searchName: string) => {
        const searchNameArray = searchName.trim().split(' ');

        if (searchNameArray.every((el) => companyNameField?.toLowerCase()?.includes(el?.trim()?.toLowerCase()))) {
            return companyNameField;
        }

        return undefined;
    };

    async searchOrganizationByName(searchName, request, errorHandler) {
        try {
            const { account_name = 'b2b_drive' } = this.state;
            const allOrganizationWallets = await request.exec(REQUESTS.GET_ACCOUNTS, {
                queryParams: {
                    account_name: encodeURIComponent(account_name),
                },
            });

            return allOrganizationWallets?.accounts?.reduce((acc: TSearchLinkArray, wallet) => {
                const match = SearchWallets.matchesSearchQuery(wallet?.data?.details?.company, searchName)
                    || SearchWallets.matchesSearchQuery(wallet?.data?.details?.comment, searchName)
                    || SearchWallets.matchesSearchQuery(wallet?.data?.comment, searchName);

                if (match) {
                    acc.push({
                        link: `#/b2b/search-wallets?${SEARCH_ID_TYPE.ORGANIZATION_ID}=${wallet.data.id}`,
                        linkText: `${match} - ${wallet.data?.id}`,
                    });
                }

                return acc;
            }, []);
        } catch (error) {
            errorHandler(error);

            return null;
        }
    }

    async searchWalletLimit(searchId, request, errorHandler): Promise<ISearchWalletsResults | null> {
        try {
            const result: ISearchWalletsResults = {
                organizations: {},
                walletLimits: [],
                walletAccounts: {},
            };

            const walletLimit = await request.exec(REQUESTS.GET_WALLETS, {
                queryParams: {
                    access_tags: organizationCreatorTag,
                    account_name: encodeURIComponent(searchId),
                },
            });
            result.walletLimits = walletLimit.accounts;

            for (const account of result.walletLimits) {
                if (account.meta?.parent_id) {
                    const organization = await request.exec(REQUESTS.GET_ACCOUNTS, {
                        queryParams: {
                            account_id: account.meta?.parent_id,
                        },
                    });

                    result.organizations = {
                        ...result.organizations,
                        [organization.id]: organization,
                    };
                }

                if (account.name) {
                    const { accounts } = await request.exec(REQUESTS.GET_ACCOUNTS, {
                        queryParams: {
                            account_name: encodeURIComponent(account?.name),
                        },
                    });
                    result.walletAccounts = {
                        ...result.walletAccounts,
                        [account.name]: accounts,
                    };
                }
            }

            return result;
        } catch (error) {
            errorHandler(error);

            return null;
        }
    }

    async searchOrganizationsById(searchId, request, errorHandler) {
        try {
            const result: ISearchWalletsResults = {
                organizations: {},
                walletLimits: [],
                walletAccounts: {},
            };

            const organization = await request.exec(REQUESTS.GET_ACCOUNTS, {
                queryParams: {
                    account_id: searchId,
                },
            });

            result.organizations = {
                ...result.organizations,
                [organization.id]: organization,
            };

            const { accounts } = await request.exec(REQUESTS.GET_WALLETS, {
                queryParams: {
                    parent_id: searchId,
                    access_tags: organizationCreatorTag,
                },
            });
            result.walletLimits = accounts;

            for (const account of result.walletLimits) {
                if (account.name) {
                    const { accounts } = await request.exec(REQUESTS.GET_ACCOUNTS, {
                        queryParams: {
                            account_name: encodeURIComponent(account?.name),
                        },
                    });

                    result.walletAccounts = {
                        ...result.walletAccounts,
                        [account.name]: accounts,
                    };
                }
            }

            return result;
        } catch (error) {
            errorHandler(error);

            return null;
        }
    }

    searchFromSearchParams() {
        const queries = new URLSearchParams(this.props.location.search);

        for (const idKey in SEARCH_ID_TYPE) {
            const searchId = queries.get(SEARCH_ID_TYPE[idKey]);

            if (searchId) {
                this.setState({
                    searchQuery: searchId,
                    searchNameType: null,
                    searchIdType: SEARCH_ID_TYPE[idKey] === SEARCH_ID_TYPE.PARENT_ID
                        ? SEARCH_ID_TYPE.ORGANIZATION_ID
                        : SEARCH_ID_TYPE[idKey],
                }, async () => {
                    await this.searchById();
                });
            }
        }
    }

    componentDidMount() {
        this.searchFromSearchParams();
    }

    componentDidUpdate(prevProps: Readonly<ISearchWalletsProps>) {
        if (this.props.location.search !== prevProps.location.search) {
            this.setState({
                searchQuery: '',
                searchByIdResults: null,
                searchByNameResults: null,
                searchIdType: null,
                searchNameType: null,
            }, () => {
                this.searchFromSearchParams();
            });
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            error,
            isSearching,
            searchQuery,
            searchByIdResults,
            searchIdType,
            searchNameType,
            searchByNameResults,
            account_name,
        } = this.state;

        const searchOptions = [
            { text: 'Клиент из кабинета b2b', value: 'b2b_drive' },
            { text: 'Корп. клиент такси', value: 'corp_yataxi' },
            { text: 'Корп. клиент Драйва', value: 'corp_drive' },
        ];

        if (isLocalhost()) {
            searchOptions.push({ text: 'Тестовый кошель фронта', value: 'test_b2b_frontend' });
        }

        return <>
            {error
                ? <SimpleError error={error}/>
                : null
            }
            <div className={styles.searchWalletsHeader}>
                <div className={styles.searchWalletsBars}>
                    <WalletsSearchBar searchId={searchQuery}
                                      onChangeSearch={this.onChangeSearch.bind(this)}
                                      searchIdType={searchIdType}
                                      expectedSearchIdType={SEARCH_ID_TYPE.WALLET_LIMIT_ID}
                                      showOr/>
                    <WalletsSearchBar searchId={searchQuery}
                                      onChangeSearch={this.onChangeSearch.bind(this)}
                                      searchIdType={searchIdType}
                                      expectedSearchIdType={SEARCH_ID_TYPE.ORGANIZATION_ID}
                                      showOr/>
                    <div>
                        <WalletsSearchBar searchId={searchQuery}
                                          onChangeSearch={this.onChangeSearchByName.bind(this)}
                                          searchIdType={searchNameType}
                                          expectedSearchIdType={SEARCH_NAME_TYPE.ORGANIZATION_NAME}/>
                        <span className={styles.organizationNameOption}>
                            {searchNameType === SEARCH_NAME_TYPE.ORGANIZATION_NAME
                                && <Select initialValues={[account_name]}
                                           placeholder={'Укажите тип организации'}
                                           options={searchOptions}
                                           onSelect={(account_name) => {
                                               this.setState({
                                                   account_name: account_name as unknown as string,
                                               });
                                           }}/>}
                        </span>
                    </div>
                </div>
                <Button className={styles.searchButton}
                        disabled={!searchQuery.length}
                        isLoading={isSearching}
                        onClick={this.searchButton.bind(this)}>
                    Поиск
                </Button>
            </div>
            {searchByIdResults && Object.keys(searchByIdResults?.organizations)?.length
                ? <WalletsSearchResults request={this.request}
                                        searchResults={searchByIdResults}/>
                : searchByNameResults?.length
                    ? <ByNameSearchResults searchResults={searchByNameResults}/>
                    : <NoInformation/>
            }
        </>;
    }
}
