import produce from 'immer';
import * as React from 'react';

import { GetSessions, IGetSessionsPropsExport } from '../../../decorators/GetSessions';
import { ITagHistoryWithPropsExport, TagHistoryWith } from '../../../decorators/TagHistoryWith';
import { Dict, TAG_ACTION } from '../../../types';
import { COINS_PER_RUBLE } from '../../constants';
import { SessionHistoryInfoHandler } from '../../models/session';
import { TagRecordHandler } from '../../models/tag';
import { UIStatusTypes } from '../../ui';
import { Button, ButtonTypes } from '../../ui/Button';
import { Cross } from '../../ui/Cross';
import FormatDate from '../../ui/FormatDate';
import { Window } from '../../ui/FullModal';
import { Input } from '../../ui/Input';
import { Link } from '../../ui/Link';
import { NoInformation } from '../../ui/NoInformation';
import Select from '../../ui/Select';
import * as tblStyle from '../../ui/Table/index.css';
import TextArea from '../../ui/TextArea';
import { NAVIGATION } from '../../utils/navigation';
import { Request2 } from '../../utils/request';
import { rubs } from '../../utils/rubs';
import ShortSessionInfoBlock from '../Chats/ShortSessionInfoBlock';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import * as style from './index.css';
import { BONUS_REQUESTS, REQUESTS } from './request';

const DEFAULT_NUMDOC = 50;

interface IBonusControlDialogProps extends IGetSessionsPropsExport, ITagHistoryWithPropsExport {
    onClose: () => void;
    userId: string;
    sendMessage?: (text: string) => void;
    from?: string[];
    sessionId?: string;
    chat_id?: string;
}

interface IBonusControlDialogState {
    selectedRideIndex: number | null;

    comment: string;
    amount: number | string;
    links: string[];
    tag_name: string;
    priority: number;

    bonusIsSending: boolean;
    error: Error | null;
}

@GetSessions({ numdoc: DEFAULT_NUMDOC })
@TagHistoryWith()
export class BonusControlDialog extends React.Component<IBonusControlDialogProps, IBonusControlDialogState> {
    state: IBonusControlDialogState = {
        selectedRideIndex: null,

        comment: '',
        amount: 0,
        links: [''],
        tag_name: '',
        priority: 0,

        bonusIsSending: false,
        error: null,
    };

    request = new Request2({ requestConfigs: BONUS_REQUESTS });

    componentDidMount(): void {
        const { getSessions, userId, sessionId, getTagsHistory } = this.props;

        getSessions?.({
            user_id: userId,
            session_id: sessionId,
        });

        getTagsHistory?.({ object_id: userId });
    }

    componentDidUpdate(prevProps: Readonly<IBonusControlDialogProps>, prevState: Readonly<IBonusControlDialogState>) {
        if (this.state.selectedRideIndex !== null) {
            const amount = this.props?.tagHistoryData?.bonusesMeta?.[this.state.selectedRideIndex ?? 0]?.amount;

            if (prevState.selectedRideIndex !== this.state.selectedRideIndex && amount) {
                this.setState({ amount });
            }
        }

        if (!Object.keys(prevProps.sessionsData ?? {}).length && Object.keys(this.props.sessionsData ?? {}).length) {
            this.setState({ selectedRideIndex: 0 });
        }
    }

    onChange(field: string, value) {
        this.setState(produce(this.state, draft => {
            if (field === 'tag_name') {
                const tag = this.props.tagHistoryData?.descriptions?.records?.[value];
                const descriptionAmount = tag?._meta?.amount;
                const priority = tag?.default_priority ?? 0;

                draft['amount'] = descriptionAmount && +(descriptionAmount / COINS_PER_RUBLE).toFixed(0)
                    || this.state.amount;
                draft['priority'] = priority;
            }

            if (field === 'selectedRideIndex') {
                value = value === '' ? null : value;
            }

            draft[field] = value ?? null;
        }));
    }

    getTagMeta() {
        const { tagHistoryData } = this.props;
        const { tag_name } = this.state;

        return tagHistoryData?.descriptions?.records[tag_name]?._meta;
    }

    send() {
        const { sessionsData, from, userId, chat_id, sendMessage, onClose } = this.props;
        const { selectedRideIndex, links, amount, comment, tag_name, priority } = this.state;

        const data = sessionsData ?? {};
        const session_id = SessionHistoryInfoHandler.getSessionId.call(data, selectedRideIndex) || null;
        const links_props = from?.map((uri) => ({
            type: 'other',
            uri,
        })) ?? [];
        const links_state = links?.filter((uri) => uri !== '')
            ?.map((uri) => ({
                type: 'other',
                uri,
            })) ?? [];

        const is_realtime = this.getTagMeta()?.is_realtime;

        const body = {
            amount: +amount * COINS_PER_RUBLE,
            car_number: SessionHistoryInfoHandler.getCarNumber.call(data, selectedRideIndex),
            comment: comment,
            object_id: userId,
            tag: tag_name,
            priority: +priority,
            links: [...links_state, ...links_props],
            ...is_realtime && { topic_link: chat_id },
            chat_topic: chat_id,
        };

        if (session_id) {
            body['session_id'] = session_id;
        }

        this.setState(produce(this.state, draft => {
            draft.bonusIsSending = true;
        }), () => {
            this.request.exec(REQUESTS.SEND_BONUS, { body })
                .then(() => {
                    this.setState(produce(this.state, draft => {
                        draft.bonusIsSending = false;
                        draft.error = null;
                    }), () => {
                        amount && sendMessage
                        && sendMessage(amount.toString());
                        onClose();
                    });
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.bonusIsSending = false;
                        draft.error = error;
                    }));
                });
        });
    }

    resetInput() {
        if (this.state.amount === 0) {
            this.setState({ amount: '' });
        }
    }

    addLink() {
        this.setState((prev) => ({
            links: [...prev.links, ''],
        }));
    }

    changeLink(index, value) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1, value);

            return {
                links,
            };
        });
    }

    deleteLink(index) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1);

            return {
                links,
            };
        });
    }

    render() {
        const {
            tagHistoryData, onClose, sessionsData, sessionError, sessionOptions, from,
            tagHistoryError, tagHistoryIsLoading,
        } = this.props;
        const { tag_name, selectedRideIndex, links = [], comment, amount, error, bonusIsSending } = this.state;

        const currentOption = tagHistoryData?.bonusesOptions?.find?.(option => option.value === tag_name);
        const rideInitialValues = selectedRideIndex !== null && selectedRideIndex !== undefined
            ? [selectedRideIndex]
            : null;

        const isSendButtonDisabled = !this.state.amount || this.state.comment === ''
            || selectedRideIndex === null || selectedRideIndex === undefined || !tag_name;

        const bonus_history = this.props?.tagHistoryData?.bonuses?.records
            ?.filter((record: Dict<any>) => record.action === TAG_ACTION.REMOVE);

        return <Window onClose={onClose.bind(this)}
                       title={'Отправка бонусов пользователю'}
                       className={style.bonus_dialog}
                       error={sessionError}>
            <div className={style.bonus}>
                <div className={`${style.bonus_panel} ${style.bonus_panel_form}`}>
                    <div className={style.form}>
                        <Select placeholder={'Причина'}
                                options={tagHistoryData?.bonusesOptions ?? []}
                                onSelect={this.onChange.bind(this, 'tag_name')}
                                initialValues={tag_name ? [tag_name] : null}
                                disabled={tagHistoryIsLoading}
                                required/>
                        <div className={style.option_description}>
                            {currentOption?.description}
                        </div>
                        <Select placeholder={'Поездка'}
                                options={sessionOptions ?? []}
                                onSelect={this.onChange.bind(this, 'selectedRideIndex')}
                                initialValues={rideInitialValues}
                                disabled={this.props.sessionIsLoading}
                                required/>
                        {
                            selectedRideIndex !== null && selectedRideIndex !== undefined
                                ? <ShortSessionInfoBlock sessionInfo={sessionsData ?? {}}
                                                         index={selectedRideIndex ?? 0}/>
                                : null
                        }
                        {
                            links?.map((link, index) => {
                                return <LinksInput key={index}
                                                   link={link}
                                                   onChange={this.changeLink.bind(this, index)}
                                                   onDelete={this.deleteLink.bind(this, index)}/>;
                            })
                        }
                        <Button className={style.addLinkButton}
                                onClick={this.addLink.bind(this)}>
                            {links.length ? 'Еще одну ссылку' : 'Добавить ссылку'}
                        </Button>
                        <TextArea placeholder={'Комментарий'}
                                  value={comment}
                                  onChange={this.onChange.bind(this, 'comment')}
                                  required/>
                        <Input value={amount}
                               placeholder={'Величина бонуса в рублях ₽'}
                               onChange={this.onChange.bind(this, 'amount')}
                               onClick={this.resetInput.bind(this)}/>

                        {from?.map((uri, index) => (
                            <Input status={{ type: UIStatusTypes.warning }}
                                   key={`uri-${index}`}
                                   value={uri}
                                   placeholder={'Ссылка на обращение (readonly)'}
                                   disabled
                                   onChange={() => undefined}/>
                        ))}
                    </div>
                    {
                        error
                            ? <SimpleError error={error}/>
                            : null
                    }
                    <div>
                        <Button onClick={this.send.bind(this)}
                                className={style.right}
                                colorType={ButtonTypes.positive}
                                isLoading={bonusIsSending}
                                disabled={isSendButtonDisabled}>
                            Отправить
                        </Button>
                    </div>
                </div>
                <div className={`${style.bonus_panel} ${style.bonus_panel_history}`}>
                    {
                        tagHistoryError ? <SimpleError error={tagHistoryError}/> : null
                    }
                    {tagHistoryIsLoading
                        ? <Spin/>
                        : !bonus_history?.length
                            ? <NoInformation/>
                            : <table className={tblStyle.table}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>дата</th>
                                        <th>тег</th>
                                        <th colSpan={3}/>
                                    </tr>
                                </thead>
                                <tbody>
                                    {bonus_history
                                        ?.map((bonusHistoryItem: Dict<any>, index) => {
                                            const rootAmount = bonusHistoryItem.tag_details
                                            && bonusHistoryItem.tag_details.amount
                                            && rubs(bonusHistoryItem.tag_details.amount);
                                            const descriptionsAmount = tagHistoryData?.descriptions
                                            && (
                                                tagHistoryData.descriptions.records[bonusHistoryItem.tag_name]
                                                && tagHistoryData.descriptions.records[bonusHistoryItem.tag_name]._meta
                                                && tagHistoryData.descriptions
                                                    .records[bonusHistoryItem.tag_name]._meta.amount
                                                && rubs(tagHistoryData.descriptions
                                                    .records[bonusHistoryItem.tag_name]._meta.amount)
                                                || tagHistoryData.descriptions.deprecated[bonusHistoryItem.tag_name]
                                                && tagHistoryData.descriptions
                                                    .deprecated[bonusHistoryItem.tag_name]._meta
                                                && tagHistoryData.descriptions
                                                    .deprecated[bonusHistoryItem.tag_name]._meta.amount
                                                && rubs(tagHistoryData.descriptions
                                                    .deprecated[bonusHistoryItem.tag_name]._meta.amount)
                                            );

                                            const amount = rootAmount
                                            || descriptionsAmount;

                                            const links = TagRecordHandler.getTagLinks.call(bonusHistoryItem);

                                            return <>
                                                <tr key={index}>
                                                    <td>{++index}</td>
                                                    <td><FormatDate withSecond
                                                                    value={TagRecordHandler.getTagTimestamp
                                                                        .call(bonusHistoryItem)}/>
                                                    </td>
                                                    <td>
                                                        {TagRecordHandler.getTagDisplayName.call(bonusHistoryItem)
                                                    || TagRecordHandler.getTagName.call(bonusHistoryItem)}
                                                    </td>
                                                    <td>{amount}</td>
                                                    <td className={style.comment}>
                                                        {TagRecordHandler.getTagComment.call(bonusHistoryItem)}
                                                    </td>
                                                </tr>
                                                {links.length
                                                    ? <BonusLinks links={links}/>
                                                    : null
                                                }
                                            </>;
                                        })}
                                </tbody>
                            </table>
                    }
                </div>
            </div>
        </Window>;
    }
}

export const BonusLinks = ((props: { links: string[]; colSpan?: number; className?: string }) => {
    const DEFAULT_COLSPAN = 3;

    return <>
        {props.links.map((uri, index) => (
            <tr className={`${tblStyle.table} ${index < props.links.length - 1 ? tblStyle.smallPadding : ''} `
                + `${props.className || ''}`}
                key={index}>
                <td colSpan={props.colSpan || DEFAULT_COLSPAN}/>
                <td>{index === 0 ? <b>Ссылки:</b> : ''}</td>
                <td colSpan={1}>
                    <Link href={uri}
                          target={'_blank'}>
                        {uri && uri.includes(NAVIGATION.ROUTES.CHATS.uri) ? 'чат' : uri}
                    </Link>
                </td>
            </tr>
        ))}
    </>;
});

export const LinksInput = (props: { link: string; onChange: () => void; onDelete: () => {} }) => {
    return (
        <div className={style.addLinkContent}>
            <Cross className={style.delete}
                   onClick={props.onDelete.bind(null)}/>
            <Input value={props.link}
                   onChange={props.onChange.bind(null)}
                   placeholder={'ссылки вставлять сюда'}/>
        </div>
    );
};
