import * as React from 'react';

import { Dict } from '../../../../types';
import { DEDICATED_FLEET_TAG } from '../../../constants';
import CarInfo, { CarInfoHandler } from '../../../models/car';
import { CopyIcon } from '../../../ui/CopyIcon';
import { CarStatus } from '../../../ui/Status';
import { CarIndicatorsBlock } from '../../../utils/carIndicators';
import CarNumber from '../../CarNumber';
import { BadgeContainer, BadgeItem } from '../../Clients/Achievements2/Badge';
import { SPECIAL_FLEET_TAG, TELEMATICS_LAG } from '../constants';
import { CarLabel, SPECIAL_FLEET_BADGE } from '../helpers';
import * as style from '../index.css';
import DedicatedFleetBadge from './DedicatedFleetBadge';

interface ICarCardHeaderProps {
    carInfo: Dict<any>;
    isLoading: boolean;
}

const ENGINE_TAGS_TO_SHOW = ['check_engine_blocker', 'check_engine_critical', 'check_engine_minor', 'check_engine'];
const ENGINE_TAG = 'check_engine';

export const CarCardHeader = React.memo((props: ICarCardHeaderProps) => {
    const { carInfo = {} } = props;
    const usage = CarInfoHandler.getUsage.call(carInfo);
    const telematics_lag = carInfo?.tags?.filter(el => el.tag === TELEMATICS_LAG)[0];
    const special_fleet_tag = carInfo?.tags?.find(el => el.tag === SPECIAL_FLEET_TAG);

    const carRenderTags = carInfo?.tags && carInfo.tags
        .filter((tag) => {
            return tag.priority;
        })
        .sort((a, b) => {
            if (a.priority === b.priority) {
                return a.display_name.localeCompare(b.display_name);
            }

            return b.priority - a.priority;
        });

    const engineTags = carInfo?.tags
        ?.filter?.(el => ENGINE_TAGS_TO_SHOW.some(item => item === el.tag))
        ?.sort((a, b) => {
            return a.display_name.localeCompare(b.display_name);
        });

    const dedicatedFleetTag = props.carInfo?.tags?.find?.(el => el.tag === DEDICATED_FLEET_TAG);
    const hasLabelsRow = carRenderTags?.length || telematics_lag || engineTags?.length;

    const views: any = carInfo?.views?.[0] || {};
    const segmentLabel = views.segment;
    const segmentUrl = views.image_small_url;

    return <div className={`${style.car_header}`}>
        <div className={style.main_title}>
            <h2>{CarInfoHandler.getModelName.call(carInfo)}</h2>

            <CopyIcon text={carInfo?.id}
                      title={'Скопировать ID в буфер'}
                      internalTooltipId={'copy_car_id'}/>

            {carInfo?.number
                ? <CarNumber carInfo={carInfo as typeof CarInfo}/>
                : null
            }

            {carInfo?.status
                ? <CarStatus text={CarInfoHandler.getStatus.call(carInfo)}/>
                : null
            }

            {usage
                ? <span className={style.usage}>/ {usage}</span>
                : null
            }
        </div>

        {dedicatedFleetTag
            ? <DedicatedFleetBadge dedicatedFleetTag={dedicatedFleetTag} />
            : null
        }

        <div className={style.header_details}>
            {hasLabelsRow
                ? <div className={`${style.info_row} ${style.short_margin}`}>
                    {telematics_lag
                        ? <CarLabel key={telematics_lag.tag_id}
                                    label={telematics_lag.display_name}
                                    className={style.telematics_lag}/>
                        : null
                    }

                    {carRenderTags.map(tag => {
                        return <CarLabel className={style.label_tag}
                                         key={tag.tag_id}
                                         label={`${tag.priority}: ${tag.display_name}`}/>;
                    })
                    }

                    {engineTags?.map(tag => {
                        const stylePostfix = tag.tag.substr(ENGINE_TAG.length + 1);
                        const styleName = `engine_${stylePostfix}`;

                        return <CarLabel className={`${style.label_tag} ${style[styleName]}`}
                                         key={tag.tag_id}
                                         label={tag.display_name}/>;
                    })}
                </div>
                : null
            }

            <CarIndicatorsBlock carInfo={carInfo}
                                wrapperStyle={style.info_row}
                                itemStyle={style.telematics_info_item}
                                iconStyle={style.telematics_icon}
                                hasTooltip={true}
                                allItems={true}/>

            {carInfo?.sf?.length
                ? <div className={style.info_row}>
                    <BadgeContainer>
                        {carInfo?.sf?.sort((a, b) => a.index - b.index).map((item) => {
                            return <BadgeItem url={item.public_icon}
                                              title={`${item.display_name}\n${item.comment}`}
                                              key={item.index}
                                              enabled/>;
                        })
                        }
                        {
                            segmentUrl &&
                            <BadgeItem url={segmentUrl}
                                       title={segmentLabel}
                                       rowClassName={style.segment}
                                       key={segmentLabel}
                                       enabled/>
                        }

                        {special_fleet_tag
                            ? SPECIAL_FLEET_BADGE()
                            : null
                        }
                    </BadgeContainer>
                </div>
                : null
            }
        </div>
    </div>;
});
