import * as React from 'react';

import { Dict } from '../../../../types';
import { CUSTOM_TAGS, EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { CAR_ACTIONS, CarInfoHandler, IConfig } from '../../../models/car';
import { UserInfoHandler } from '../../../models/user';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { QuickActionButton } from '../../../ui/QuickActionButton';
import { IBlockRules } from '../../../utils/IBlockRules';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import { ytLogs } from '../../../utils/sendLogs';
import { buttonLocationDetails, buttonNameDetails } from '../../../utils/sendLogs/eventTypes/buttonDetails';
import { IStore } from '../../App/store';
import { SidebarButtons } from '../../Clients/CCView/CCSidebar/SidebarButtons';
import { Copy } from '../../Copy';
import DocForQueuePicker from '../../Documents/DocForQueuePicker';
import FastTagsView from '../../FastTagsView';
import IncidentForm from '../../IncidentForm';
import { IFastTagItem } from '../../InterfaceAdminConfig/FastTags/schema';
import TagModal from '../../TagModal';
import { OBJECT_TYPES } from '../../TagModal/component';
import { INSURANCE_TYPES } from '../../VinSimView/UploadingInsuranceModal/types';
import AnyCommand from '../AnyCommand';
import CarLessor from '../CarLessor';
import { StartSessionModal } from '../CarStartSessionModal/component';
import { NISSAN_LEAF } from '../constants';
import * as style from '../index.css';
import { LongtermFeedButtonWrapper } from '../LongtermFeedButtonWrapper';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import SendCarToChat from '../SendCarToChat';
import ShowDamagesLikeClient from '../ShowDamagesLikeClient';
import { CompensationButton } from "./CompensationButton";
import LeafControls from './LeafControls';

interface ICarCardSidebarProps extends IStore, IBlockRules {
    carInfo: Dict<any>;
    getData: () => void;
    showPassword: () => void;
    passwordTelematics: string | null;
}

interface ICarCardSidebarState {
    question: string;
    confirmIsOpen: boolean;
    accept: () => void;
    isWorking: boolean;
    isTagModalOpen: boolean;
    isDocsQueuePickerOpen: boolean;
    isIncidentNewFormOpen: boolean;
    isStartSessionModalOpen: boolean;
    tagType: string[];
    error: Error | null;
    fastTag: IFastTagItem | null;
}

export class CarCardSidebar extends React.Component<ICarCardSidebarProps, ICarCardSidebarState> {
    log = ytLogs.getInstance();
    state: ICarCardSidebarState = {
        question: '',
        confirmIsOpen: false,
        accept: () => {
        },
        isWorking: false,
        isTagModalOpen: false,
        isDocsQueuePickerOpen: false,
        isIncidentNewFormOpen: false,
        isStartSessionModalOpen: false,
        tagType: [],
        error: null,
        fastTag: null,
    };
    request = new Request2({ requestConfigs });
    ls = new LS();

    onClose() {
        this.setState({
            confirmIsOpen: false,
            isWorking: false,
        });
    }

    activeSession() {
        const carInfo: Dict<any> = this.props.carInfo;
        const saas = carInfo && carInfo.session_info
            && carInfo.session_info.meta
            && !carInfo.session_info.meta.finished && carInfo.session_info.meta.session_id || false;
        const saasUser = carInfo && carInfo.user
            && carInfo.user.id;

        return { saas, saasUser };
    }

    defaultConfig = {
        finishOrder: false, callback: () => {
        }, ignoreError: false, question: '',
    };

    carControlHandle(actionType: string, config: IConfig = this.defaultConfig) {
        const carInfo: Dict<any> = this.props.carInfo;
        const { car_id } = config;

        const accept = this.handleCarAction.bind(this, config.actionName === REQUESTS.FINISH_CAR_SESSION
            ? config.user_id
            : car_id || carInfo.id,
        actionType, config);

        return accept();
    }

    resetHead() {
        const carInfo: Dict<any> = this.props.carInfo;

        return this.request.exec(REQUESTS.RESET_HEAD, {
            queryParams: { car_id: carInfo.id },
        });
    }

    handleCarAction(id: string, type: string, config: IConfig = this.defaultConfig): Promise<any> {
        const {
            callback, actionName = REQUESTS.CAR_ACTION_STAFF
            , evolution_mode = 'default',
        } = config;

        const headers = actionName === REQUESTS.FINISH_CAR_SESSION ? { UserIdDelegation: id } : {};
        const session_id = this.props?.carInfo?.session_info?.meta.session_id;

        return this.request.exec(actionName, {
            headers: headers,
            queryParams: { command: type, car_id: id, evolution_mode, session_id },
        })
            .then(() => {
                if (callback) {
                    callback();
                }
            });
    }

    openTagModal(tagType: string[], fastTag: IFastTagItem | null) {
        this.setState({
            tagType,
            isTagModalOpen: true,
            fastTag,
        });
    }

    closeTagModal() {
        this.setState({ isTagModalOpen: false });
    }

    openDocsQueuePicker() {
        this.setState({ isDocsQueuePickerOpen: true });
    }

    closeDocsQueuePicker() {
        this.setState({ isDocsQueuePickerOpen: false });
    }

    openIncidentNewForm() {
        this.setState({ isIncidentNewFormOpen: true });
    }

    closeIncidentNewForm() {
        this.setState({ isIncidentNewFormOpen: false });
    }

    openStartSessionModal(isOpen) {
        this.setState({ isStartSessionModalOpen: isOpen });
    }

    render() {
        const { carInfo } = this.props;
        const speed = carInfo
            && carInfo.telematics
            && carInfo.telematics.speed || 0;

        const user = CarInfoHandler.getUserInfo.call(carInfo);
        const name = UserInfoHandler.getPrintName.call(user);
        const phone = UserInfoHandler.getPhone.call(user);
        const sessionMeta = CarInfoHandler.getSessionMeta.call(carInfo);
        const session = sessionMeta
            ? `${FormatDate({
                value: sessionMeta.start * ONE_SECOND,
                withSecond: true,
            }).props.children} - ${FormatDate({
                value: sessionMeta.finish * ONE_SECOND,
                withSecond: true,
            }).props.children}`
            : null;
        const imei = CarInfoHandler.getImei.call(carInfo);
        const sts = CarInfoHandler.getSts.call(carInfo);
        const vin = CarInfoHandler.getVin.call(carInfo);
        const insurance_provider = CarInfoHandler.getInsuranceProvider.call(carInfo);
        const osago_link = CarInfoHandler.getOsagoLink.call(carInfo);
        const car_id = CarInfoHandler.getId.call(carInfo);

        const questionDetails = <div className={style.confirm_details}>
            <div>Авто: <strong className={style.strong}>{carInfo.number} </strong>({carInfo.model_id})</div>
            <div>Пользователь: <strong className={style.strong}>{phone}</strong> {name}</div>
            <div>Заказ: {session}</div>
        </div>;
        const showScooterControls = this.ls.envIsScooter();
        const isTestingEnv = this.ls.envIsBasicTesting();
        const showLeafControls = carInfo.model_id === NISSAN_LEAF;

        return <div className={style.car_side_bar}>
            <table className={style.sidebar_table}>
                <tbody>
                    <tr>
                        <td className={style.key}>КП</td>
                        <td>{CarInfoHandler.isManualGearbox.call(carInfo) ? 'Механика' : 'Автомат'}</td>
                    </tr>
                    {this.props.BlockRules && this.props.BlockRules.support
                        ? <>

                            <tr>
                                <td className={style.key}>IMEI</td>
                                <td>
                                    <Copy text={imei}
                                          internalTooltipId={'imei'}>
                                        {imei}
                                    </Copy>
                                </td>
                            </tr>
                            <tr>
                                <td className={style.key}>СТС</td>
                                <td>
                                    <Copy text={sts}
                                          internalTooltipId={'sts'}>
                                        {sts}
                                    </Copy>
                                </td>
                            </tr>
                            <tr>
                                <td className={style.key}>VIN</td>
                                <td>
                                    <Copy text={vin}
                                          internalTooltipId={'vin'}>
                                        {vin}
                                    </Copy>
                                </td>
                            </tr>
                            <tr>
                                <td className={style.key}>Страховая</td>
                                <td>{
                                    INSURANCE_TYPES[insurance_provider?.toUpperCase()]?.text || EMPTY_DATA
                                }</td>
                            </tr>
                        </>
                        : null}
                    <tr className={style.tracks}>
                        <td/>
                        <td>
                            <Link href={`#/tracks?car_id=${this.props.carInfo && this.props.carInfo.id}&status=`}>
                            Треки
                            </Link>

                            {this.props.BlockRules?.DownloadInsurance
                                ? <Link href={osago_link} target={'_blank'} className={style.osago}> ОСАГО</Link>
                                : null
                            }
                        </td>
                    </tr>
                    <tr>
                        <td colSpan={2}><ShowDamagesLikeClient carInfo={this.props.carInfo}/></td>
                    </tr>
                    <tr>
                        <td title={'Пароль инженерного сервера'}
                            onClick={this.props?.showPassword.bind(this)}
                            className={`${style.password} ${style.key}`}>
                            <div className={style.password_title}>
                            Пароль <span className={`${style.password_icon} ${this.props.passwordTelematics ? style.open : style.close}`}/>
                            </div>
                        </td>
                        <td className={style.password_value}>
                            {this.props.passwordTelematics || '*******'}
                        </td>
                    </tr>
                    <tr>
                        <CarLessor carId={car_id}/>
                    </tr>
                </tbody>
            </table>
            {showScooterControls
                ? <SidebarButtons BlockRules={this.props.BlockRules as Dict<boolean>}
                                  carControlHandle={this.carControlHandle.bind(this)}/>
                : this.props.BlockRules && this.props.BlockRules.supportFull
                    ? <>
                        <div className={style.open_close_button_container}>
                            <p>Двери</p>
                            <QuickActionButton question={<div>Открыть машину? {questionDetails}</div>}
                                               button={{
                                                   children: 'Открыть',
                                                   colorType: ButtonTypes.negative,
                                                   basic: true,
                                               }}
                                               showConfirm
                                               accept={this.carControlHandle.bind(this, CAR_ACTIONS.OPEN_DOORS)}
                                               ytLog={{
                                                   button_name: buttonNameDetails.OPEN_DOORS,
                                                   location: buttonLocationDetails.CAR_CARD,
                                               }}/>
                            <QuickActionButton question={<div>Закрыть машину? {questionDetails}</div>}
                                               button={{
                                                   children: 'Закрыть',
                                                   basic: true,
                                               }}
                                               showConfirm
                                               accept={this.carControlHandle.bind(this, CAR_ACTIONS.CLOSE_DOORS)}
                                               ytLog={{
                                                   button_name: buttonNameDetails.CLOSE_DOORS,
                                                   location: buttonLocationDetails.CAR_CARD,
                                               }}/>
                        </div>
                        {(this.props.BlockRules && this.props.BlockRules.DutySupport
                                || carInfo && carInfo.model_id !== 'skoda_rapid') &&
                            <>
                                <div className={style.open_close_button_container}>
                                    <p>Капот</p>
                                    <QuickActionButton question={<div>Открыть капот? {questionDetails}</div>}
                                                       button={{
                                                           children: 'Открыть',
                                                           colorType: ButtonTypes.negative,
                                                           basic: true,
                                                       }}
                                                       showConfirm
                                                       accept={this.carControlHandle
                                                           .bind(this, CAR_ACTIONS.YADRIVE_UNLOCK_HOOD)}
                                                       ytLog={{
                                                           button_name: buttonNameDetails.OPEN_HOOD,
                                                           location: buttonLocationDetails.CAR_CARD,
                                                       }}/>
                                    <QuickActionButton question={<div>Закрыть капот? {questionDetails}</div>}
                                                       button={{
                                                           children: 'Закрыть',
                                                           basic: true,
                                                       }}
                                                       showConfirm
                                                       accept={this.carControlHandle
                                                           .bind(this, CAR_ACTIONS.YADRIVE_LOCK_HOOD)}
                                                       ytLog={{
                                                           button_name: buttonNameDetails.CLOSE_HOOD,
                                                           location: buttonLocationDetails.CAR_CARD,
                                                       }}/>
                                </div>
                            </>
                        }
                    </>
                    : null
            }
            {
                this.props.BlockRules?.Engine && !showScooterControls &&
                <>
                    <div className={style.open_close_button_container}>
                        <p>Авто</p>
                        <QuickActionButton question={<div>Разблокировать автомобиль? {questionDetails}</div>}
                                           button={{
                                               children: 'Разблок.',
                                               colorType: ButtonTypes.negative,
                                               basic: true,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle.bind(this, CAR_ACTIONS.YADRIVE_START_OF_LEASE)}
                                           ytLog={{
                                               button_name: buttonNameDetails.UNLOCK_CAR,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                        <QuickActionButton question={<div>Блокировать автомобиль? {questionDetails}</div>}
                                           button={{
                                               children: 'Блок.',
                                               basic: true,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle.bind(this, CAR_ACTIONS.YADRIVE_END_OF_LEASE)}
                                           ytLog={{
                                               button_name: buttonNameDetails.LOCK_CAR,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                    </div>
                </>
            }

            {showLeafControls
                ? <div>
                    <LeafControls car_id={car_id}/>
                </div>
                : null
            }

            <div className={style.button_container}>
                <FastTagsView place={buttonLocationDetails.CAR_CARD}
                              addTag={this.openTagModal.bind(this, [CUSTOM_TAGS.CAR_SERVICE_TAG])}/>

                <LongtermFeedButtonWrapper user_id={user?.id}/>

                <SendCarToChat carInfo={carInfo}/>
                {
                    this.props.BlockRules?.AnyCommand
                    && <AnyCommand carInfo={carInfo} questionDetails={questionDetails}/>
                }
                {this.props.BlockRules?.ShortHorn &&
                    <QuickActionButton question={<div>Короткий гудок? {questionDetails}</div>}
                                       button={{
                                           children: 'Короткий гудок',
                                       }}
                                       showConfirm
                                       accept={this.carControlHandle.bind(this, CAR_ACTIONS.HORN)}
                                       ytLog={{
                                           button_name: buttonNameDetails.SHORT_HORN,
                                           location: buttonLocationDetails.CAR_CARD,
                                       }}/>
                }
                {
                    this.props.BlockRules?.LongHorn &&
                    <QuickActionButton question={carInfo?.telematics?.horn_dout3 == '1'
                        ? 'Выключить ДЛИННЫЙ гудок?'
                        : <div>Включить ДЛИННЫЙ гудок? {questionDetails}</div>}
                                       button={{
                                           children: carInfo?.telematics?.horn_dout3 == '1'
                                               ? 'Выключить ДЛИННЫЙ гудок?'
                                               : 'Включить ДЛИННЫЙ гудок?',
                                           colorType: carInfo?.telematics?.horn_dout3 == '1'
                                               ? ButtonTypes.warning
                                               : ButtonTypes.positive,
                                       }}
                                       showConfirm
                                       ytLog={{
                                           button_name: buttonNameDetails.LONG_HORN,
                                           location: buttonLocationDetails.CAR_CARD,
                                       }}
                                       accept={this.carControlHandle.bind(this,
                                           carInfo?.telematics?.horn_dout3 == '1'
                                               ? CAR_ACTIONS.LONG_HORN_DISABLE
                                               : CAR_ACTIONS.LONG_HORN_ENABLE, {
                                               callback: () => {
                                                   this.onClose();
                                                   this.props.getData();
                                               },
                                           },
                                       )}/>
                    || null
                }
                {
                    this.props.BlockRules && this.props.BlockRules.supportFull &&
                    <QuickActionButton question={<div>Завершить заказ? {questionDetails}</div>}
                                       button={{
                                           children: 'Завершить заказ',
                                           colorType: ButtonTypes.warning,
                                           disabled: !this.activeSession().saas,
                                       }}
                                       showConfirm
                                       accept={this.carControlHandle.bind(this, CAR_ACTIONS.ORDERFINISH, {
                                           actionName: REQUESTS.FINISH_CAR_SESSION,
                                           user_id: this.activeSession().saasUser,
                                           evolution_mode: 'default',
                                           callback: this.props.getData,
                                       })}
                                       ytLog={{
                                           button_name: buttonNameDetails.FINISH_ORDER,
                                           location: buttonLocationDetails.CAR_CARD,
                                       }}/>
                }
                {
                    this.props.BlockRules && this.props.BlockRules.Finishsession &&
                    <>
                        <QuickActionButton question={<div>Завершить заказ? {questionDetails}</div>}
                                           button={{
                                               children: 'Завершить ignore_telematic',
                                               colorType: ButtonTypes.negative,
                                               disabled: !this.activeSession().saas,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle.bind(this, CAR_ACTIONS.ORDERFINISH, {
                                               actionName: REQUESTS.FINISH_CAR_SESSION,
                                               user_id: this.activeSession().saasUser,
                                               evolution_mode: 'ignore_telematic',
                                               callback: this.props.getData,
                                           })}
                                           ytLog={{
                                               button_name: buttonNameDetails.FINISH_ORDER_IGNORE_TELEMATICS,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                        <QuickActionButton question={<div>Завершить заказ? {questionDetails}</div>}
                                           button={{
                                               children: 'Завершить force',
                                               colorType: ButtonTypes.warning,
                                               disabled: !this.activeSession().saas,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle.bind(this, CAR_ACTIONS.ORDERFINISH, {
                                               actionName: REQUESTS.FINISH_CAR_SESSION,
                                               user_id: this.activeSession().saasUser,
                                               evolution_mode: 'force',
                                               callback: this.props.getData,
                                           })}
                                           ytLog={{
                                               button_name: buttonNameDetails.FINISH_ORDER_FORCE,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                        <QuickActionButton question={speed
                            ? <div>
                                <div>Скорость Автомобиля <strong style={{ color: 'orange' }}>{speed}</strong> км/ч!!!!
                                </div>
                                <div>Действительно выполнить экстренную блокировку???</div>
                                {questionDetails}</div>
                            : <div>Экстренная блокировка? {questionDetails}</div>}
                                           button={{
                                               children: 'Экстренная блокировка',
                                               colorType: ButtonTypes.negative,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle
                                               .bind(this, CAR_ACTIONS.SCENARIO_BLE_RESET_FORCED_END_OF_LEASE)}
                                           ytLog={{
                                               button_name: buttonNameDetails.EMERGENCY_LOCK,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                    </>
                }
                {
                    this.props.BlockRules?.PoliteEngine
                    && <>
                        <QuickActionButton question={speed
                            ? <div>
                                <div>Скорость Автомобиля <strong style={{ color: 'orange' }}>{speed}</strong> км/ч!!!!</div>
                                <div>Действительно выполнить экстренную блокировку???</div>
                                {questionDetails}</div>
                            : <div>Деликатная блокировка? {questionDetails}</div>}
                                           button={{
                                               children: 'Деликатная блокировка',
                                               colorType: ButtonTypes.warning,
                                               basic: true,
                                           }}
                                           showConfirm
                                           accept={this.carControlHandle
                                               .bind(this, CAR_ACTIONS.SCENARIO_POLITE_FORCED_END_OF_LEASE)}
                                           ytLog={{
                                               button_name: buttonNameDetails.POLITE_LOCK,
                                               location: buttonLocationDetails.CAR_CARD,
                                           }}/>
                    </>
                }
                {
                    this.props.BlockRules && this.props.BlockRules.VegaReboot &&
                    <QuickActionButton question={<div>Перегрузить вегу? {questionDetails}</div>}
                                       showConfirm
                                       button={{
                                           children: 'Перегрузить вегу',
                                       }}
                                       accept={this.carControlHandle.bind(this, CAR_ACTIONS.SCENARIO_POLITE_RESTART)}
                                       ytLog={{
                                           button_name: buttonNameDetails.RESTART_VEGA,
                                           location: buttonLocationDetails.CAR_CARD,
                                       }}/>
                }
                {
                    this.props.BlockRules && this.props.BlockRules.HeadReset &&
                    <QuickActionButton question={<div>Сбросить голову? {questionDetails}</div>}
                                       button={{
                                           children: 'Сбросить голову',
                                           colorType: ButtonTypes.warning,
                                       }}
                                       showConfirm
                                       accept={this.resetHead.bind(this)}
                                       ytLog={{
                                           button_name: buttonNameDetails.RESET_HEAD,
                                           location: buttonLocationDetails.CAR_CARD,
                                       }}/>
                }
                {
                    this.props.BlockRules && this.props.BlockRules.Engine &&
                    <QuickActionButton question={<div>Включить принудительный прогрев? {questionDetails}</div>}
                                       button={{
                                           children: 'Включить прогрев',
                                           basic: true,
                                       }}
                                       showConfirm
                                       accept={this.carControlHandle.bind(this, CAR_ACTIONS.YADRIVE_WARMING)}
                                       ytLog={{ button_name: buttonNameDetails.WARM_CAR }}/>
                }
                {this.props.BlockRules && this.props.BlockRules.support
                    ? <Button colorType={ButtonTypes.positive}
                              ytLog={{
                                  button_name: buttonNameDetails.SERVICE_TAG,
                                  location: buttonLocationDetails.CAR_CARD,
                              }}
                              onClick={this.openTagModal.bind(this, [CUSTOM_TAGS.CAR_SERVICE_TAG], null)}>
                        Сервисная задача
                    </Button>
                    : null}
                <Button colorType={ButtonTypes.warning}
                        basic
                        ytLog={{
                            button_name: buttonNameDetails.SHOW_DOCUMENTS,
                            location: buttonLocationDetails.CAR_CARD,
                        }}
                        onClick={this.openDocsQueuePicker.bind(this)}>Документы</Button>
                {
                    this.props.BlockRules && this.props.BlockRules.NewIncidentForm
                        ? <Button onClick={this.openIncidentNewForm.bind(this)}
                                  colorType={ButtonTypes.negative}
                                  ytLog={{
                                      button_name: buttonNameDetails.INCIDENT_NEW,
                                      location: buttonLocationDetails.SESSION_CARD,
                                  }}>Инцидент</Button>
                        : null
                }

                <CompensationButton car={carInfo}/>

                {isTestingEnv
                    ? <Button colorType={ButtonTypes.positive}
                              basic={true}
                              onClick={this.openStartSessionModal.bind(this, true)}>
                        Начать движение
                    </Button>
                    : null
                }

            </div>
            {this.state.isTagModalOpen
                ? <TagModal objectId={{ type: OBJECT_TYPES.CAR, id: this.props.carInfo.id && this.props.carInfo.id }}
                            tagType={this.state.tagType}
                            initialData={this.state.fastTag || null}
                            onClose={this.closeTagModal.bind(this)}/>
                : null}
            {this.state.isDocsQueuePickerOpen
                ? <DocForQueuePicker initValues={{
                    car_id: this.props.carInfo && this.props.carInfo.id,
                    comment: this.props.carInfo && this.props.carInfo.number,
                }}
                                     onClose={this.closeDocsQueuePicker.bind(this)}/>
                : null}
            {this.state.isIncidentNewFormOpen
                ? <IncidentForm carInfo={this.props.carInfo} onClose={this.closeIncidentNewForm.bind(this)}/>
                : null}
            {this.state.isStartSessionModalOpen
                ? <StartSessionModal onClose={this.openStartSessionModal.bind(this, false)} carId={car_id}/>
                : null
            }
        </div>;
    }

}
