import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import { UIStatusTypes } from '../../../ui';
import { Button } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import Select from '../../../ui/Select';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CarContext } from '../context';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import * as style from './index.css';

const MODEL_CODE = 'model_code';

interface ICarInfo {
    [key: string]: string;
}

interface INewDataElement {
    old?: string;
    new: string;
}

interface INewData {
    [key: string]: INewDataElement;
}

interface IHistoryItem {
    event_id: string;
    timestamp: number;
    user_id: string;
    newData?: INewData;
    object: ICarInfo;
}

interface ICarModel {
    [key: string]: any;
}

interface ICarEditorProps {
    carId: string;
}

interface ICarEditorState {
    [key: string]: any;

    isLoading: boolean;
    loadingError: Error | null;
    upsertingError: Error | null;
    updateHistory: IHistoryItem[];
    carInfo: ICarInfo;
    carInfoInitial: ICarInfo;
    editedFields: string[];
    isConfirmOpened: boolean;
    invalidFields: string;
    carModelsItems: any[];
}

export default class CarEditor extends React.Component<ICarEditorProps, ICarEditorState> {
    state: ICarEditorState = {
        isLoading: false,
        loadingError: null,
        updateHistory: [] as IHistoryItem[],
        carInfo: {} as ICarInfo,
        carInfoInitial: {} as ICarInfo,
        editedFields: [],
        isConfirmOpened: false,
        invalidFields: '',
        carModelsItems: [],
        [`${MODEL_CODE}_INIT`]: [],
        [MODEL_CODE]: [],
        upsertingError: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ICarEditorProps>): void {
        if (this.props.carId !== prevProps.carId) {
            this.getData();
        }
    }

    getData() {
        this.setState({ isLoading: true }, () => {
            Promise.all([
                this.request.exec(REQUESTS.GET_CAR_UPSERT_HISTORY, { queryParams: { car_id: this.props.carId } }),
                this.request.exec(REQUESTS.GET_MODELS),
            ])
                .then((response) => {
                    const history = response[0];
                    const models: ICarModel[] = response[1] && response[1].car_models || [];
                    const carInfoSaas = this.context;

                    const updateHistory = history && history.history;
                    updateHistory && updateHistory.length
                    && updateHistory.reduce((prev: IHistoryItem, curr: IHistoryItem) => {

                        const prevObject = prev.object;
                        const currObject = curr.object;

                        Object.keys(currObject).forEach((currKey: string) => {
                            const currValue = currObject[currKey];
                            const prevValue = prevObject[currKey];

                            if (currValue !== prevValue) {
                                const diff = { old: prevValue, new: currValue };
                                if (curr.newData) {
                                    curr.newData[currKey] = diff;
                                } else {
                                    curr.newData = { [currKey]: diff };
                                }
                            }
                        });

                        return curr;
                    }, { object: {} });

                    updateHistory.reverse();

                    const carModelsItems = models
                        && models
                            .map(carModel => {
                                return { text: carModel.name, value: carModel.code, description: carModel.code };
                            })
                            .sort((a, b) => a.text.localeCompare(b.text))
                        || [];

                    const car = carInfoSaas;
                    const carInfo = {
                        id: car.id,
                        vin: car.vin,
                        model_code: car.model_id,
                        status: car.status,
                        number: car.number,
                        registration_id: car.registration_id,
                        responsible_picker: car.responsible_picker,
                        imei: car.imei,
                        osago_mds_key: car.osago_mds_key,
                        registration_mds_key: car.registration_mds_key,
                        major_app_mds_key: car.major_app_mds_key,
                    };

                    this.setState({
                        isLoading: false,
                        carInfo,
                        [`${MODEL_CODE}_INIT`]: carInfo.model_code,
                        [MODEL_CODE]: carInfo.model_code,
                        updateHistory,
                        carModelsItems,
                        carInfoInitial: carInfo,
                        editedFields: [],
                    });
                })
                .catch(loadingError => {
                    this.setState({
                        isLoading: false,
                        loadingError,
                    });
                });
        });
    }

    upsertCarInfo() {
        const { id } = this.state.carInfo;
        const data = this.state.editedFields.reduce((res: { [key: string]: any }, editedField: string) => {
            res[editedField] = this.state.carInfo?.[editedField] ?? '';

            return res;
        }, { id });

        return this.request.exec(REQUESTS.UPSERT_CAR_INFO,
            { body: data, queryParams: { force: this.state.invalidFields } });
    }

    onAcceptConfirm() {
        this.setState({ isWorking: true }, () => {
            this.upsertCarInfo()
                .then(() => {
                    this.setState({ isWorking: false });
                    this.closeConfirm();
                }).catch((upsertingError: any) => {
                    const invalidFields = upsertingError.response && upsertingError.response.data
                    && upsertingError.response.data.errors[0] && upsertingError.response.data.errors[0].field;
                    if (invalidFields) {
                        this.setState({ isWorking: false, invalidFields, upsertingError });
                    } else {
                        this.setState({ isWorking: false, upsertingError });
                    }
                });
        });
    }

    openConfirm() {
        this.setState({ isConfirmOpened: true, invalidFields: '', fullErrorData: null });
    }

    closeConfirm() {
        this.setState({ isConfirmOpened: false, fullErrorData: null, upsertingError: null });
    }

    onInputChange(key: string, value: string) {
        this.updateCarInfo(key, value);
    }

    onSelectChange(type: string, value: string) {
        this.setState({ [type]: [value] });
        this.updateCarInfo(type, value);
    }

    updateCarInfo(key: string, value: string) {
        const carInfo: ICarInfo = Object.assign({}, this.state.carInfo);
        carInfo[key] = value;

        const editedFields: string[] = [];

        Object.keys(carInfo).forEach((key: string) => {
            const carInfoInitial: ICarInfo = Object.assign({}, this.state.carInfoInitial);
            if (carInfoInitial[key] !== carInfo[key]) {
                editedFields.push(key);
            }
        });

        this.setState({ carInfo, editedFields });
    }

    render() {
        return <div className={style.car_garage}>
            <h2>Редактор автомобиля</h2>
            {this.state.loadingError
                ? <SimpleError error={this.state.loadingError} data={{ label: 'Ошибка при загрузке редактора' }}/>
                : this.state.isLoading
                    ? <Spin/>
                    : <>

                        <div className={style.editor}>
                            <EditorContainer carInfo={this.state.carInfo}
                                             editedFields={this.state.editedFields}
                                             modelItems={this.state.carModelsItems}
                                             modelCode={this.state[`${MODEL_CODE}_INIT`]}
                                             onSelectChange={this.onSelectChange.bind(this)}
                                             onInputChange={this.onInputChange.bind(this)}
                                             openConfirm={this.openConfirm.bind(this)}/>
                        </div>
                        <HistoryView history={this.state.updateHistory} getData={this.getData.bind(this)}/>
                    </>}
            {this.state.isConfirmOpened
                ? <Confirm question={
                    <div>
                        <div>Обновить информацию о машине?</div>
                        {
                            this.state.invalidFields && <div>Некорректные поля: {this.state.invalidFields}</div>
                        }
                    </div>}
                           onClose={this.closeConfirm.bind(this)}
                           isWorking={this.state.isWorking}
                           error={this.state.upsertingError}
                           accept={this.onAcceptConfirm.bind(this)}/>
                : null}
        </div>;
    }
}

CarEditor.contextType = CarContext;

interface IDiffElementProps {
    diff: [string, INewDataElement];
}

const DiffElement = (props: IDiffElementProps) => {
    const { diff } = props;

    const key = diff[0];
    const oldValue = diff[1].old;
    const newValue = diff[1].new;

    return <div className={style.diff}>
        <span className={style.key}>{key}</span>:
        {oldValue
        && <span>
            <span className={`${style.value} ${style.old}`}>{oldValue}</span>
            → </span>}
        <span className={`${style.value} ${style.new}`}>{newValue}</span>
    </div>;
};

interface IHistoryRowProps {
    historyItem: IHistoryItem;
    index: number;
}

const HistoryRow = (props: IHistoryRowProps) => {
    const { historyItem, index } = props;

    return <tr className={style.diff_row}>
        <td>{index + 1}</td>
        <td>
            <FormatDate value={historyItem.timestamp * ONE_SECOND}/>
        </td>
        <td>
            <Link href={`#/clients/${historyItem.user_id}/info`}>Пользователь</Link>
        </td>
        <td>
            {historyItem.newData
            && Object.entries(historyItem.newData).map((diff) => {
                return <DiffElement key={diff[0]} diff={diff}/>;
            })
            || <div className={style.diff}>
                <span className={style.no_diff}>Изменений не было</span>
            </div>}
        </td>
    </tr>;
};

interface IHistoryViewProps {
    history: IHistoryItem[];
    getData: () => void;
}

const HistoryView = (props: IHistoryViewProps) => {
    const { history, getData } = props;

    return <div>
        <h3>История изменений (<Link onClick={getData}>обновить</Link>)</h3>
        <div className={style.history_container}>
            {history && history.length > 0
                ? <table className={`${styleTable.table} ${style.history_table}`}>

                    <tbody>
                        {history.map((item: IHistoryItem, index) => {
                            return <HistoryRow key={item.event_id} historyItem={item} index={index}/>;
                        })}
                    </tbody>
                </table>
                : <span className={style.empty_history}>История изменений пустая</span>
            }
        </div>
    </div>;
};

interface IEditorContainerProps {
    carInfo: ICarInfo;
    editedFields: string[];
    modelItems: { text: string; value: string }[];
    modelCode: string;
    onSelectChange: () => void;
    onInputChange: () => void;
    openConfirm: () => void;
}

const EditorContainer = (props: IEditorContainerProps) => {
    const { carInfo, editedFields, modelItems, modelCode, onSelectChange, onInputChange, openConfirm } = props;

    return <>
        <div className={style.editor_container}>
            {Object.keys(carInfo).length > 0
                ? Object.entries(carInfo).map((carInfoItem) => {
                    const key: string = carInfoItem[0];
                    const value = carInfoItem[1];

                    const isDisable = ['id'].includes(key);
                    const status = editedFields.includes(key) ? {
                        type: UIStatusTypes.positive,
                        text: 'Поле изменено',
                    } : null;

                    if (key === MODEL_CODE) {
                        return <Select key={key}
                                       status={status}
                                       options={modelItems}
                                       placeholder={MODEL_CODE}
                                       disabled={isDisable}
                                       initialValues={modelCode ? [modelCode] : []}
                                       containerClassName={style.input}
                                       onSelect={onSelectChange.bind(null,
                                           MODEL_CODE)}/>;
                    }

                    return <Input key={key}
                                  status={status}
                                  value={value}
                                  className={style.input}
                                  disabled={isDisable}
                                  onChange={onInputChange.bind(null, key)}
                                  placeholder={key}/>;
                })
                : <Spin/>
            }
        </div>
        <div className={style.button_container}>
            <Button onClick={openConfirm.bind(null)}
                    disabled={editedFields && editedFields.length === 0}>Изменить</Button>
            <p className={style.update_info}>Данные о ТС обновляются в течение 10 секунд</p>
        </div>
    </>;
};
