import React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import { Request2 } from '../../../utils/request';
import { isValidGUIDString } from '../../../utils/utils';
import Spin from '../../Spin';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { AttachmentItem } from './AttachmentItem';
import * as style from './index.css';
import { IAttachmentItem } from './types';

enum HISTORY_ACTION {
    add = 'add',
    remove = 'remove'
}

interface IAttachmentsHistoryProps {
    carId: string;
    onClose: () => void;
}

interface IHistoryItem {
    event_id: number;
    user_id: string;
    timestamp: number;
    object: any;
    action: HISTORY_ACTION;
    attachment?: IAttachmentHistoryItem;
}

interface IAttachmentHistoryItem {
    id: string;
    type: string;
    data: any;
}

interface IAttachmentHistory {
    attachments: IAttachmentItem[];
    history: IHistoryItem[];
}

interface IAttachmentsHistoryState {
    isHistoryLoading: boolean;
    history: IAttachmentHistory;
    error: Error | null;
    filter: string;
}

export default class AttachmentsHistory extends React.Component<IAttachmentsHistoryProps, IAttachmentsHistoryState> {
    state: IAttachmentsHistoryState = {
        isHistoryLoading: false,
        history: {} as IAttachmentHistory,
        error: null,
        filter: '',
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getHistory(this.props.carId);
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getHistory(car_id: string) {
        this.setState({ isHistoryLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ATTACHMENTS_HISTORY, { queryParams: { car_id } })
                .then(response => {
                    this.setState({
                        history: response || [],
                        isHistoryLoading: false,
                    });
                })
                .catch(error => this.setState({ error }));
        });
    }

    getActionName(id: string) {
        return this.state.history.attachments.find(el => el.id === id);
    }

    onFilterChange(filter: string) {
        this.setState({ filter });
    }

    render() {
        const { isHistoryLoading, error, filter } = this.state;
        const history = this.state.history.history;

        return <Window error={error} onClose={this.props.onClose.bind(this)} title={'История изменений'}>
            {isHistoryLoading
                ? <Spin/>
                : <div className={style.history_table}>
                    <div>
                        <Input onChange={this.onFilterChange.bind(this)}
                               value={filter}
                               placeholder={'Фильтр по истории'}
                               className={style.filter}/>
                    </div>

                    {history?.length
                        ? history.map((item, index) => {
                            const list = this.getActionName(item.object.generic_attachment_id);
                            const isVisible = !filter || (filter && JSON.stringify(list?.data)?.includes(filter));
                            const authorInfo = item.user_id
                                ? isValidGUIDString(item.user_id)
                                    ? <Link href={`#/clients/${item.user_id}/info`}>Пользователь</Link>
                                    : <span>{item.user_id}</span>
                                : <span>{EMPTY_DATA}</span>;

                            return isVisible
                                ? <div className={style.history_item} key={index}>
                                    <div className={style.head}>
                                        <div>{authorInfo}</div>
                                        <div>
                                            <FormatDate value={item.timestamp * ONE_SECOND} withSecond={true}/>
                                        </div>
                                    </div>
                                    <div className={`${style.history_action} ${style[`history_action_${item.action}`]}`}>
                                        {item.action}
                                    </div>
                                    <div className={style.json_view}>
                                        <AttachmentItem list={list as IAttachmentItem} />
                                    </div>
                                </div>
                                : null;
                        })
                        : <NoInformation/>}
                </div>
            }
        </Window>;
    }
}
