import React from 'react';

import { ONE_HOUR, ONE_SECOND } from '../../../constants';
import { Button, CancelButton, SaveButton } from '../../../ui/Button';
import { Collapse } from '../../../ui/Collapse';
import FormatDate from '../../../ui/FormatDate';
import { Confirm, Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { Request2 } from '../../../utils/request';
import { FormConstructor } from '../../FormConstructor';
import * as modalStyle from '../../Settings/Wallets/AddModal.css';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { MILEAGE_SENSOR_ID } from '../constants';
import { CarContext } from '../context';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import * as style from './index.css';
import { EDIT_TO_SCHEMA } from './schema';

interface IMaintenanceProps {
    carId: string;
}

interface IMaintenanceState {
    data: any[];
    isRemoveConfirmOpen: boolean;
    isEditModalOpen: boolean;
    checkedVin: string;
    initialData: Record<string,any>;
    formValues: Record<string,any>;
    isLoading: boolean;
    error: Error | null;
    mileageError: Error | null;
}

export default class Maintenance extends React.Component<IMaintenanceProps, IMaintenanceState> {
    state: IMaintenanceState = {
        data: [],
        isRemoveConfirmOpen: false,
        isEditModalOpen: false,
        checkedVin: '',
        initialData: {},
        formValues: {},
        isLoading: false,
        error: null,
        mileageError: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getInfo();
    }

    newContext: any = {};

    componentWillUnmount() {
        this.request.abort();
    }

    private previousContext;

    componentDidUpdate(prevProps: Readonly<IMaintenanceProps>, prevState: Readonly<IMaintenanceState>, snapshot?: any) {
        if (this.previousContext?.vin !== this.context?.vin) {
            this.getInfo();
        }

        this.previousContext = this.context;
    }

    getInfo() {
        this.setState({ isLoading: true }, () => {
            const carInfo = this.previousContext;
            const vin = carInfo.vin || '';
            if (vin) {
                this.request.exec(REQUESTS.GET_MAINTENANCE_INFO, { queryParams: { vin } })
                    .then(response => {
                        this.setState({
                            data: response.maintenance || [],
                            checkedVin: vin,
                            isLoading: false,
                        });
                    })
                    .catch(error => this.setState({ error, isLoading: false }));
            } else {
                this.setState({ error: null, isLoading: false });
            }

        });
    }

    getMileage() {
        return this.request
            .exec(REQUESTS.GET_TELEMATICS_STATE, { queryParams: { car_id: this.props.carId } })
            .then(response => {
                return response.sensors.find((el) => el.id === MILEAGE_SENSOR_ID).value;
            })
            .catch(error => this.setState({ mileageError: error }));
    }

    async onEditMaintenanceClick(isEdit: boolean) {
        let initialData;
        const { data, checkedVin } = this.state;

        if (!data.length || !isEdit) {
            const currentTime = Date.now();
            initialData = {
                vin: checkedVin,
                mileage: await this.getMileage(),
                start_date: ~~((currentTime - ONE_HOUR) / ONE_SECOND),
                ready_date: ~~(currentTime / ONE_SECOND),
            };
        } else {
            const item = data.find(el => el.vin === checkedVin);
            initialData = {
                vin: checkedVin,
                mileage: item?.mileage,
                start_date: item?.start_date,
                ready_date: item?.ready_date,
                is_intermediate: item?.is_intermediate,
            };
        }

        this.setState({ isEditModalOpen: true, initialData });
    }

    onRemoveConfirmOpen() {
        this.setState({
            isRemoveConfirmOpen: true,
        });
    }

    onEditMaintenanceClose() {
        this.setState({
            isEditModalOpen: false,
        });
    }

    onRemoveConfirmClose() {
        this.setState({
            isRemoveConfirmOpen: false,
        });
    }

    onChange(formValues) {
        this.setState({
            formValues,
        });
    }

    onSaveMaintenance() {
        this.setState({ isLoading: true }, () => {
            this.request.exec(REQUESTS.UPSERT_MAINTENANCE, { body: this.getDataForRequest() })
                .then(() => this.setState({
                    isEditModalOpen: false,
                    isLoading: false,
                }, () => this.getInfo()))
                .catch(error => this.setState({ error, isLoading: false }));
        });
    }

    getDataForRequest() {
        const { vin, is_intermediate, start_date, ready_date } = this.state.formValues;
        const mileage = +(this.state.formValues?.mileage?.toString()?.replace(',', '.')) ?? '';

        const data = {
            vin,
            is_intermediate,
            mileage,
        };

        if (start_date) {
            data['start_date'] = start_date;
        }

        if (ready_date) {
            data['ready_date'] = ready_date;
        }

        return data;
    }

    onRemoveMaintenance() {
        this.setState({ isLoading: true }, () => {
            this.state.checkedVin && this.request.exec(REQUESTS.REMOVE_MAINTENANCE, {
                queryParams: { vin: this.state.checkedVin },
            })
                .then(() => this.setState({
                    isRemoveConfirmOpen: false,
                    isLoading: false,
                }, () => this.getInfo()))
                .catch(error => this.setState({ error, isLoading: false }));
        });
    }

    render() {
        const {
            error,
            data,
            isRemoveConfirmOpen,
            isEditModalOpen,
            initialData,
            isLoading,
            mileageError,
        } = this.state;

        return <div>
            {isLoading
                ? <Spin/>
                : error
                    ? <SimpleError error={error}/>
                    : <div>
                        <div className={style.title_to}>TO</div>
                        {Boolean(data.length)
                            && <div>
                                <table className={style.maintenance_table}>
                                    <thead>
                                        <tr>
                                            <td>source</td>
                                            <td>start_date</td>
                                            <td>ready_date</td>
                                            <td>mileage</td>
                                            <td>is_intermediate</td>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {data.map((el, index) => {
                                            return <tr key={index} className={style.maintenance_item}>
                                                <td>{el.source}</td>
                                                <td><FormatDate value={el.start_date * ONE_SECOND} withSecond/></td>
                                                <td><FormatDate value={el.ready_date * ONE_SECOND} withSecond/></td>
                                                <td>{el.mileage}</td>
                                                <td>{el.is_intermediate.toString()}</td>
                                                <td>
                                                    <Link onClick={this.onEditMaintenanceClick.bind(this, true)}>
                                                        редактировать
                                                    </Link>
                                                </td>
                                                <td>
                                                    <Link onClick={this.onRemoveConfirmOpen.bind(this)}>
                                                        удалить
                                                    </Link>
                                                </td>
                                            </tr>;
                                        })}
                                    </tbody>
                                </table>
                            </div>}
                        {data.length && !data[data.length - 1].ready_date
                            ? null
                            : <Button basic
                                      onClick={this.onEditMaintenanceClick.bind(this, false)}>
                                Добавить информацию о ТО
                            </Button>}

                        {isRemoveConfirmOpen
                            ? <Confirm question={'Удалить запись?'}
                                       error={error}
                                       accept={this.onRemoveMaintenance.bind(this)}
                                       onClose={this.onRemoveConfirmClose.bind(this)}/>
                            : null
                        }
                        {isEditModalOpen
                            ? <Window onClose={this.onEditMaintenanceClose.bind(this)}
                                      className={style.overflow_init}
                                      error={error}
                                      title={'Информация о ТО'}>
                                {mileageError
                                    && <Collapse collapsed={true}
                                                 title={'Не удалось получить пробег'}>
                                        <SimpleError error={mileageError}/>
                                    </Collapse>}
                                <FormConstructor schema={EDIT_TO_SCHEMA}
                                                 onChange={this.onChange.bind(this)}
                                                 initialData={initialData}/>
                                <div className={modalStyle.buttonContainer}>
                                    <CancelButton onClick={this.onEditMaintenanceClose.bind(this)}/>
                                    <SaveButton onClick={this.onSaveMaintenance.bind(this)}/>
                                </div>
                            </Window>
                            : null
                        }
                    </div>}
        </div>;
    }
}

Maintenance.contextType = CarContext;
