import React from 'react';

import { Button } from '../../../ui/Button';
import { Confirm } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { NoInformation } from '../../../ui/NoInformation';
import { EMPTY_DATA_STATUS_CODE, Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { AttachmentItem } from './AttachmentItem';
import AttachmentsHistory from './AttachmentsHistory';
import * as style from './index.css';
import LastTyre from './LastTyre';
import Maintenance from './Maintenance';
import { IAttachmentItem } from './types';

interface ICarGarageProps {
    carId: string;
}

interface ICarGarageState {
    isLoading: boolean;
    inputValue: string;
    attachmentList: IAttachmentItem[];
    idForDetach: string;
    isDetachConfirmOpened: boolean;
    isAddConfirmOpened: boolean;
    isHistoryOpen?: boolean;
    isUpdateOsagoOpen: boolean;
    error: Error | null;
    osagoError: Error | null;
    errorWhileAttaching: Error | null;
    is204Open: boolean;
    osagoIsLoading: boolean;
    osagoResponseText: string;
}

export default class CarGarage extends React.Component<ICarGarageProps, ICarGarageState> {
    state: ICarGarageState = {
        attachmentList: [],
        inputValue: '',
        isLoading: false,
        idForDetach: '',
        isDetachConfirmOpened: false,
        isHistoryOpen: false,
        isAddConfirmOpened: false,
        isUpdateOsagoOpen: false,
        error: null,
        osagoError: null,
        errorWhileAttaching: null,
        is204Open: false,
        osagoIsLoading: false,
        osagoResponseText: '',
    };
    request = new Request2({ requestConfigs });

    componentDidMount(): void {
        this.getAttachmentList();
    }

    componentDidUpdate(
        prevProps: Readonly<ICarGarageProps>,
        prevState: Readonly<ICarGarageState>,
        snapshot?: any,
    ): void {
        if (prevProps.carId !== this.props.carId) {
            this.getAttachmentList();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getAttachmentList() {
        this.setState({ isLoading: true }, () => {
            this.request.exec(REQUESTS.GET_ATTACHMENTS_LIST, {
                queryParams: { car_id: this.props.carId },
                withCode: true, // returning array with response data and response status code
            })
                .then(response => {
                    this.setState({
                        attachmentList: response[0]?.attachments || [],
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({ error, isLoading: false });
                });
        });
    }

    onInputChange(val) {
        this.setState({ inputValue: val });
    }

    showAddAttachment(isAddConfirmOpened: boolean) {
        this.setState({ isAddConfirmOpened });
    }

    onClickAddAttachment() {
        this.state.inputValue && this.request.exec(REQUESTS.ASSIGN_ATTACHMENT, {
            queryParams: { car_id: this.props.carId },
            body: { device_code: this.state.inputValue },
        })
            .then(() => {
                this.setState({
                    inputValue: '',
                    isAddConfirmOpened: false,
                }, () => this.getAttachmentList());
            })
            .catch(errorWhileAttaching => this.setState({ errorWhileAttaching }));

    }

    onDetachSelect(id: string) {
        this.setState({ isDetachConfirmOpened: true, idForDetach: id });
    }

    onClickDetach() {
        this.state.idForDetach && this.request
            .exec(REQUESTS.REMOVE_ATTACHMENTS, { queryParams: { attachment_id: this.state.idForDetach } })
            .then(() => {
                this.setState({ isDetachConfirmOpened: false });
                this.getAttachmentList();
                this.closeDetachConfirm();
            })
            .catch((errorWhileAttaching) => {
                this.setState({ errorWhileAttaching, idForDetach: '' });
            });
    }

    updateOsago() {
        this.setState({
            osagoIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.UPDATE_OSAGO, { body: { car_id: this.props.carId }, withCode: true })
                .then((response) => {
                    this.setState({
                        osagoResponseText: response[1] === EMPTY_DATA_STATUS_CODE
                            ? 'Документ от major совпадает с текущим. Обновления не произошло!'
                            : 'Документ обновлён!',
                        is204Open: true,
                        isUpdateOsagoOpen: false,
                        osagoIsLoading: false,
                    }, () => this.getAttachmentList());

                })
                .catch((osagoError) => {
                    this.setState({ osagoError, osagoIsLoading: false });
                });
        });

    }

    closeDetachConfirm() {
        this.setState({ isDetachConfirmOpened: false });
    }

    showHistory(isHistoryOpen: boolean) {
        this.setState({ isHistoryOpen });
    }

    showUpdateOsago(isUpdateOsagoOpen: boolean) {
        this.setState({ isUpdateOsagoOpen, osagoError: null });
    }

    show204(status) {
        this.setState({
            is204Open: status,
        });
    }

    render() {
        const {
            isLoading, inputValue, isDetachConfirmOpened, isAddConfirmOpened, attachmentList, error,
            errorWhileAttaching, osagoError, isHistoryOpen, isUpdateOsagoOpen, is204Open,
        } = this.state;

        return <div className={style.main}>
            {isLoading ? <Spin/> :
                error
                    ? <SimpleError error={error}/>
                    : <>
                        <Maintenance carId={this.props.carId}/>
                        <LastTyre carId={this.props.carId}/>

                        <div className={style.labels}>
                            <Input value={inputValue}
                                   onChange={this.onInputChange.bind(this)}
                                   placeholder={'Новое устройство'}/>
                            <Button onClick={this.showAddAttachment.bind(this, true)}>Добавить новое устройство</Button>
                            <Button onClick={this.showHistory.bind(this, true)}>Показать историю измений</Button>
                            <Button onClick={this.showUpdateOsago.bind(this, true)}>Обновить osago</Button>
                        </div>

                        {attachmentList.length
                            ? attachmentList.map(list => {
                                return <AttachmentItem list={list}
                                                       onClick={this.onDetachSelect.bind(this)}
                                                       key={list.id}/>;
                            })
                            : <NoInformation/>
                        }
                    </>
            }
            {
                isDetachConfirmOpened
                    ? <Confirm question={'Удалить информацию об элементе?'}
                               error={errorWhileAttaching}
                               accept={this.onClickDetach.bind(this)}
                               onClose={this.closeDetachConfirm.bind(this)}/>
                    : null
            }
            {
                isAddConfirmOpened
                    ? <Confirm question={'Добавить новое устройство?'}
                               error={errorWhileAttaching}
                               className={style.confirm}
                               accept={this.onClickAddAttachment.bind(this)}
                               onClose={this.showAddAttachment.bind(this, false)}/>
                    : null
            }
            {
                isHistoryOpen
                    ? <AttachmentsHistory carId={this.props.carId}
                                          onClose={this.showHistory.bind(this, false)}/>
                    : null
            }
            {
                isUpdateOsagoOpen
                    ? <Confirm onClose={this.showUpdateOsago.bind(this, false)}
                               question={'Обновить osago?'}
                               error={osagoError}
                               className={style.confirm}
                               isWorking={this.state.osagoIsLoading}
                               accept={this.updateOsago.bind(this)}/>
                    : null
            }
            {
                is204Open
                    ? <Confirm onClose={this.show204.bind(this, false)}
                               error={null}
                               className={style.confirm}
                               question={this.state.osagoResponseText}
                               accept={this.show204.bind(this, false)}/>
                    : null
            }
        </div>;
    }
}
