import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { GetSessions } from '../../../../decorators/GetSessions';
import { TagDescriptions } from '../../../../decorators/TagDescriptions';
import ChildChairIcon from '../../../../svg-components/childChair.component.svg';
import { Dict } from '../../../../types';
import { EMPTY_DATA, FREE_DELEGATION, ONE_SECOND } from '../../../constants';
import { CAR_STATUS, CarInfoHandler } from '../../../models/car';
import { UserInfoHandler } from '../../../models/user';
import { DelegationIcon } from '../../../ui/DelegationIcon';
import FormatDate, { dateLag, durationBetween } from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { IBlockRules } from '../../../utils/IBlockRules';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import LS from '../../../utils/localStorage/localStorage';
import { Request2 } from '../../../utils/request';
import { copySessionText } from '../../Chats/utils';
import { MiniMapCatchError } from '../../Clients/CCView/CCViewSession/miniMapCatchError';
import { OfferInfoTooltip } from '../../Clients/SessionView/component';
import { ErrorSource, logError } from '../../Content/initErrorCounter';
import { Copy } from '../../Copy';
import { ErrorView } from '../../GlobalErrorCatchHandler';
import MiniMap from '../../MiniMap';
import { SessionBadges } from '../../SessionsList/SessionItem/badges';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CarContext } from '../context';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import * as style from './index.css';
import { NextOrder } from './NextOrder';

const IS_BATTERY_LOW = 1;
const IS_VISIBLE_MARK = 1;

interface ICarInfoProps extends IBlockRules {
    carId: string;
}

interface ICarInfoState {
    isLoading: boolean;
    loadingError: Error | null;
    carInfo: Dict<any>;
    userInfo: Dict<any>;
    errorView: Error | null;
}

export class CarInfo extends React.Component<ICarInfoProps, ICarInfoState> {
    state: ICarInfoState = {
        isLoading: false,
        loadingError: null,
        carInfo: {},
        userInfo: {},
        errorView: null,
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ICarInfoProps>): void {
        if (this.state.loadingError) {
            return;
        }

        if (this.props.carId !== prevProps.carId) {
            this.getData();
        }

        if (this.context !== this.state.carInfo) {
            this.getData();
        }
    }

    getData() {
        const carInfo = this.context;

        this.setState({
            carInfo,
        }, () => {
            const user_id = carInfo && carInfo.user && carInfo.user.id;

            user_id && this.request.exec(REQUESTS.GET_USER_INFO, { queryParams: { user_id: user_id } })
                .then(userInfo => {
                    this.setState({
                        userInfo,
                        isLoading: false,
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError });
                });
        });
    }

    getUserTariffCopyText() {
        const carInfo = this.state.carInfo || {};
        const { number: carNumber = '' } = carInfo;
        const user = carInfo && carInfo.user && carInfo.user;

        return copySessionText(user, carNumber);
    }

    componentDidCatch(errorView: Error, errorInfo: React.ErrorInfo): void {
        logError(errorView, ErrorSource.ERROR_BOUNDARY, errorInfo);

        this.setState({
            errorView,
        });
    }

    render() {
        const { carInfo, errorView } = this.state;

        const isDelegated = carInfo?.tags?.find(el => el.tag === FREE_DELEGATION) || false;
        const showSessionInfo = carInfo.session_info;

        const userName = `${UserInfoHandler.getPrintNameWithoutPn.call(carInfo.user)}`;
        const carStatus = CarInfoHandler.getStatus.call(carInfo);
        let cardColor = `${style.block} ${style.session_info}`;
        switch (carStatus) {
        case CAR_STATUS.old_state_reserv:
        case CAR_STATUS.old_state_reserv_testing:
            cardColor += ` ${style.reserved}`;
            break;
        case CAR_STATUS.old_state_accept:
            cardColor += ` ${style.accepted}`;
            break;
        case CAR_STATUS.old_state_parkin:
            cardColor += ` ${style.parking}`;
            break;
        case CAR_STATUS.available:
            cardColor += ` ${style.available}`;
            break;
        case CAR_STATUS.old_state_riding:
            cardColor += ` ${style.riding}`;
            break;
        default:
            break;
        }

        return <div className={style.car_info}>
            {
                this.state.loadingError &&
                <SimpleError error={this.state.loadingError} data={{ label: 'Ошибка при загрузке данных' }}/>
            }
            {carInfo
                ? <>
                    <div>
                        <div className={style.telematics}>
                            <Doors carInfo={this.state.carInfo}/>
                            {carInfo.telematics && carInfo.telematics.mark_1 !== null &&
                            <Equipment telematics={carInfo.telematics}/>}
                        </div>
                    </div>
                    {showSessionInfo && <div className={style.session_data}>
                        {carInfo.session_info
                            ? <div className={style.block}>
                                <table>
                                    <tbody>
                                        <tr>
                                            <td>
                                                <strong>Начало</strong>:
                                            </td>
                                            <td>
                                                <FormatDate withSecond={true}
                                                            value={carInfo.session_info && carInfo.session_info.meta
                                                        && carInfo.session_info.meta.start * ONE_SECOND}/>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>
                                                <strong>Конец</strong>:
                                            </td>
                                            <td>
                                                <FormatDate withSecond={true}
                                                            value={carInfo.session_info && carInfo.session_info.meta
                                                        && carInfo.session_info.meta.finish * ONE_SECOND}/>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>
                                                <strong>Длительность</strong>:
                                            </td>
                                            <td>
                                                {
                                                    carInfo.session_info
                                                    && carInfo.session_info.meta
                                                    && carInfo.session_info.meta.start
                                                && durationBetween([
                                                    carInfo.session_info
                                                    && carInfo.session_info.meta
                                                    && carInfo.session_info.meta.finish,
                                                    carInfo.session_info
                                                    && carInfo.session_info.meta
                                                    && carInfo.session_info.meta.start,
                                                ])
                                                || EMPTY_DATA
                                                }
                                            &nbsp;
                                                {
                                                    carInfo.session_info
                                                    && carInfo.session_info.meta
                                                    && carInfo.session_info.meta.session_id
                                                &&
                                                <Link href={`#/session/${carInfo.session_info.meta.session_id}?back=${encodeURIComponent(location.hash)}`}>сессия</Link>
                                                || EMPTY_DATA
                                                }
                                            </td>
                                        </tr>
                                        <tr>
                                            <td colSpan={2}>
                                                <BadgesForCarCard session_id={carInfo?.session_info?.meta?.session_id}/>
                                            </td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                            : null}
                        {carInfo.session_info
                            ?
                            <div className={cardColor}>
                                <Copy text={this.getUserTariffCopyText()}
                                      internalTooltipId={'user_offer'}>
                                    <table>
                                        <tbody>
                                            {
                                                carInfo.user
                                                    ? <tr>
                                                        <td colSpan={2}>
                                                            <Link href={`#/clients/${carInfo.user.id}/sessions`}>
                                                                {userName}
                                                                {isDelegated && <DelegationIcon/>}
                                                            </Link>
                                                        </td>
                                                    </tr>
                                                    : EMPTY_DATA
                                            }
                                            {
                                                carInfo?.session_info?.session?.specials
                                            && <>
                                                <tr>
                                                    <td>Тариф:</td>
                                                    <td>
                                                        <strong>
                                                            <OfferInfoTooltip BlockRules={this.props.BlockRules}
                                                                              className={style.offer_tooltip}
                                                                              current_offer={CarInfoHandler
                                                                                  .getSessionCurrentOffer.call(carInfo)}
                                                                              total={CarInfoHandler
                                                                                  .getSessionTotalPrice.call(carInfo)}
                                                                              index={'0'}/>
                                                        </strong>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td>Итого:</td>
                                                    <td>
                                                        <strong>
                                                            {CarInfoHandler.getSessionTotalPrice.call(carInfo)}
                                                        </strong>
                                                    </td>
                                                </tr>
                                            </>
                                            }
                                        </tbody>
                                    </table>
                                </Copy>
                            </div>
                            : null
                        }
                        {(carInfo?.tags?.length > 0 ?? false) && (
                            <NextOrder data={carInfo.tags}
                                       className={style.block}
                                       carNumber={carInfo && carInfo.number}/>
                        )}
                    </div>}
                </>
                : <Spin/>
            }
            <div className={style.minimap}>
                {errorView
                    ? <ErrorView error={errorView}/>
                    : <MiniMapCatchError>
                        <MiniMap carInfo={this.state.carInfo}
                                 userRealtimeData={this.state.userInfo && this.state.userInfo.realtime_data}
                                 showExpandButton/>
                    </MiniMapCatchError>
                }
            </div>
        </div>;
    }
}

CarInfo.contextType = CarContext;

const Doors = (props: { carInfo: Dict<any> }) => {
    const ls = new LS();
    const showDoors = !ls.envIsScooter();

    return showDoors
        ? <div className={style.doors_items}>
            <div className={style.doors_item}>
                <div className={style.door_name}>Левая передняя</div>
                <DoorsLag carInfo={props.carInfo} name={'is_front_left_door_open'}/>
            </div>
            <div className={style.doors_item}>
                <div className={style.door_name}>Правая передняя</div>
                <DoorsLag carInfo={props.carInfo} name={'is_front_right_door_open'}/>
            </div>

            <div className={style.doors_item}>
                <div className={style.door_name}>Багажник</div>
                <DoorsLag carInfo={props.carInfo} name={'is_trunk_open'}/>
            </div>
            <div className={style.doors_item}>
                <div className={style.door_name}>Левая задняя</div>
                <DoorsLag carInfo={props.carInfo} name={'is_rear_left_door_open'}/>
            </div>
            <div className={style.doors_item}>
                <div className={style.door_name}>Правая задняя</div>
                <DoorsLag carInfo={props.carInfo} name={'is_rear_right_door_open'}/>
            </div>
            <div className={style.doors_item}>
                <div className={style.door_name}>Капот</div>
                <DoorsLag carInfo={props.carInfo} name={'is_hood_open'}/>
            </div>
        </div>
        : null;
};

const DoorsLag = (props: { carInfo: Dict<any>; name: string }) => {
    const telematics = props.carInfo && props.carInfo.telematics || {};

    const isExist: boolean = telematics[props.name];

    return <div className={`${style.door_lag} ${!isExist ? style.empty : ''}`}>
        {
            isExist
                ? dateLag(telematics[`${props.name}_updated_at`] * ONE_SECOND)
                : EMPTY_DATA
        }
    </div>;
};

interface IEquipment {
    telematics: {
        [key: string]: any;
        low_battery_marks: number | null;
        visible_marks: number | null;
        mark_1_updated_at: number;
        visible_marks_updated_at: number;
        mark_1: number | null;
    };
}

export const Equipment = (props: IEquipment) => {
    const {
        low_battery_marks,
        visible_marks,
        mark_1_updated_at,
        visible_marks_updated_at,
        mark_1,
    } = props.telematics || {} as IEquipment;
    const isLowBattery = low_battery_marks === IS_BATTERY_LOW;
    const isVisibleMarks = visible_marks === IS_VISIBLE_MARK;

    const charStyle = !isVisibleMarks ? style.chair__icon_scrounge : isLowBattery ? style.chair__icon_discharged : '';

    return <div className={style.car_info__equipment}>
        <div className={style.equipment__title}>ОБОРУДОВАНИЕ</div>
        <div className={style.equipment__chair}>
            <ChildChairIcon data-tip
                            data-for={'chair'}
                            className={`${style.chair__icon} ${charStyle}`}/>
            <ReactTooltip id={'chair'} place="right">
                <div>{mark_1}</div>
                {isLowBattery &&
                <>
                    <div>{'Разрядилось'}</div>
                    <FormatDate value={mark_1_updated_at * ONE_SECOND}/>
                </>
                }
            </ReactTooltip>

            {!isVisibleMarks &&
            <div className={style.chair__time}>
                <div className={style.chair__text}>Пропало</div>
                <FormatDate value={visible_marks_updated_at * ONE_SECOND}/>
            </div>}
        </div>
    </div>;
};

@GetSessions({ numdoc: 1 })
@TagDescriptions()
class BadgesForCarCard extends React.Component<any, any> {
    getData() {
        if (this.props.session_id) {
            this.props?.getTags?.();
            this.props?.getSessions?.({ session_id: this.props.session_id });
        }
    }

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<any>, snapshot?: any) {
        if (this.props.session_id !== prevProps.session_id) {
            this.getData();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<any>, nextState: Readonly<any>, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps);
    }

    render() {
        return <div>
            {this.props.sessionsData ? <SessionBadges index={0}
                                                      sessionsData={this.props.sessionsData}
                                                      internalTooltip
                                                      tagsObject={this.props.tagsObject}/> : null}
        </div>;
    }
}
