import moment from 'moment';
import * as React from 'react';

import { ISearchControls, WithSearchControls } from '../../../../decorators/SearchControls';
import { Button } from '../../../ui/Button';
import { Input } from '../../../ui/Input';
import { Request2 } from '../../../utils/request';
import AcceptancePhotos, { IAcceptancePhotos } from '../../AcceptancePhotos/component';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CAR_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface ICarTagsPhotosProps extends ISearchControls {
    carId: string;
}

interface ICarTagsPhotosState {
    data?: IAcceptancePhotos;
    error: Error | null;
    isLoading: boolean;
    tag_names: string;
}

@WithSearchControls({
    since: moment().subtract(1, 'w'),
    until: moment(),
})
export class CarTagsPhotos extends React.Component<ICarTagsPhotosProps, ICarTagsPhotosState> {
    state: ICarTagsPhotosState = {
        isLoading: false,
        tag_names: '',
        error: null,
    };
    request = new Request2({ requestConfigs: CAR_REQUESTS });

    componentDidMount() {
        this.getData();
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {
            const { tag_names } = this.state;
            const { since, until, carId } = this.props;
            this.request.exec(REQUESTS.GET_TAG_PHOTOS, {
                queryParams: {
                    car_id: carId,
                    since,
                    until,
                    tag_names,
                },
            })
                .then((response) => {
                    const images = response && response.images;
                    let data: IAcceptancePhotos = { groups: [], links: [] };
                    if (images) {
                        data = {
                            groups: Object.keys(images).map((key) => {
                                const links = images[key].reduce((_p, item) => {
                                    return [..._p, ...item?.images?.map(el => {
                                        return {
                                            link: Boolean(response.host && (el.path || item.path))
                                                ? response.host + '/' + (el.path || item.path)
                                                : el.url || item.url,
                                            timestamp: item.timestamp,
                                            session_id: null,
                                            user_id: item.history_user_id,
                                        };
                                    })];
                                }, []);

                                return {
                                    links,
                                    tags: [key],
                                    timestamp: 0,
                                };
                            }),
                            links: [],
                        };
                    }

                    this.setState({
                        data,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    onChange(value) {
        this.setState({
            tag_names: value,
        });
    }

    render() {
        const { data, tag_names, isLoading, error } = this.state;

        return (
            <>
                <div className={styles.input_container}>
                    <Input placeholder={'Введите теги'}
                           onChange={this.onChange.bind(this)}
                           value={tag_names}/>
                    <Button onClick={this.getData.bind(this)}>Загрузить</Button>
                </div>
                {error ? <SimpleError error={error}/> : isLoading ? <Spin/> :
                    <AcceptancePhotos data={data}/>
                }
            </>
        );
    }
}
