import React, { useState } from 'react';

import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes, CancelButton } from '../../../../ui/Button';
import { Collapse2 } from '../../../../ui/Collapse2';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import { CAR_REQUESTS, REQUESTS } from '../../request';
import { ITag } from '../types';
import style from './index.css';

interface IAssignPerformerModalProps {
    onClose: () => void;
    tag: ITag;

    techRoles?: string;
}

export const AssignPerformerModal = React.memo((props: IAssignPerformerModalProps) => {
    const roles = props.techRoles ?? '';
    const request = new Request2({ requestConfigs: CAR_REQUESTS });

    const [user, selectUser] = useState(null);
    const [performError, setPerformError] = useState<any>(null);
    const [performLoading, setPerformLoading] = useState<boolean>(false);

    const options = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_USERS_BY_ROLES,
            requestOptions: {
                queryParams: { roles },
            },
        };
    }, [roles]);

    function getSelectOptions(response): IOptionInfo[] {
        const usersArray = response.report ?? [];
        const options: IOptionInfo[] = [];

        usersArray.forEach(user => {
            const desc = `email: ${UserInfoHandler.getEmail.call(user)}, phone: ${UserInfoHandler.getPhone.call(user)}`;
            options.push({
                value: UserInfoHandler.getId.call(user),
                text: UserInfoHandler.getPrintName.call(user),
                description: desc,
            });
        });

        return options;
    }

    const [
        isLoading,
        response,
        error,
        makeRequest,
    ] = useRequestHandler<IOptionInfo[]>(request, options, getSelectOptions);

    React.useEffect(() => {
        makeRequest();
    }, []);

    function assignPerformer() {
        if (props.tag.tag_id && user) {
            setPerformLoading(true);
            request.exec(REQUESTS.SET_PERFORMER, {
                body: {
                    worker_id: user,
                    tag_id: props.tag.tag_id,
                },
            })
                .then(() => {
                    props.onClose();
                    setPerformLoading(false);
                })
                .catch(performError => {
                    setPerformError(performError);
                    setPerformLoading(false);
                });
        }
    }

    const isPerformPermissionError = performError?.data?.error_details?.debug_message.includes('no_permissions');
    const performPermissionTitle = 'Ошибка:( Возможно, у пользователя недостаточно прав. ';

    const errorComponent = <SimpleError error={performError}/>;

    const performErrorComponent = isPerformPermissionError
        ? <Collapse2 title={performPermissionTitle} children={errorComponent} headerClassname={style.error}/>
        : errorComponent;

    return <Window onClose={props.onClose.bind(null)}
                   title={'Назначить исполнителя'}
                   error={error}
                   className={style.main}>

        {performError
            ? performErrorComponent
            : null
        }

        <Select options={response ?? []}
                onSelect={selectUser.bind(null)}
                isLoading={isLoading}
                placeholder={'Техник'}/>

        <div className={coreStyle.button_container}>
            <CancelButton onClick={props.onClose.bind(null)}/>

            <Button onClick={assignPerformer.bind(null)}
                    isLoading={performLoading}
                    colorType={ButtonTypes.positive}
                    disabled={!user}>
                Назначить
            </Button>
        </div>

    </Window>;
});
