import * as React from 'react';
import { Route, Switch } from 'react-router';

import { Dict } from '../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { CarInfoHandler } from '../../../../models/car';
import { AdminUserReducerState } from '../../../../reducers/adminUserReducer';
import { Button } from '../../../../ui/Button';
import { Collapse } from '../../../../ui/Collapse';
import FormatDate from '../../../../ui/FormatDate';
import { Confirm } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import * as styleTable from '../../../../ui/Table/index.css';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import TagModal from '../../../TagModal';
import { OBJECT_TYPES, Operations } from '../../../TagModal/component';
import { TagsDeepHistory2 } from '../../../TagsDeepHistory2/component';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../../request';
import AssignPerformerModal from '../AssignPerformerModal';
import { ComplaintHistoryModal } from '../ComplaintHistoryModal/component';
import { MoreTagInfoModal } from '../MoreTagInfoModal';
import { RepairTagInfoModal } from '../RepairTagInfoModal';
import { TelematicStatus } from '../TelematicStatus';
import { ICarFormattedTag, IComplaintTag, ITag, TagActions } from '../types';
import { CarCurrentTagsTable } from './table';

const CAR_REPAIR_TAG_NAME = 'car_repair_tag';

interface ICarCurrentTagsProps {
    allTagsHash: Dict<ITag>;
    carId: string;
    iface_tag_implementations: Dict<any>;
    AdminUser: AdminUserReducerState | undefined;
    importantCarTags?: Dict<any>;
    history: any;
}

interface ICarCurrentTagsState {
    carTagsIsLoading: boolean;
    error: Error | null;
    tags: ICarFormattedTag[];
    propositions: any[];
    currentRepairTagId: string;
    currentRepairTag: ICarFormattedTag | null;
    isRejectConfirmOpen: boolean;
    isWorking: boolean;
    rejectError: Error | null;
    proposition_users: any;
    confirmIsOpen: boolean;
    question: any;
    accept: any;
    confirmError: Error | null;
    addEvolution: boolean;
    evolutionTag: any;
    isEditTag: boolean;
    editingTag: ITag | null;
    showArchive: boolean;
    complaintHistoryTag: IComplaintTag | null;
    assignPerformerTag: ITag | null;
}

export class CarCurrentTags extends React.Component<ICarCurrentTagsProps, ICarCurrentTagsState> {
    state: ICarCurrentTagsState = {
        carTagsIsLoading: false,
        error: null,
        tags: [],
        propositions: [],
        currentRepairTagId: '',
        currentRepairTag: null,
        isRejectConfirmOpen: false,
        isWorking: false,
        rejectError: null,
        proposition_users: null,
        confirmIsOpen: false,
        question: null,
        accept: null,
        confirmError: null,
        addEvolution: false,
        evolutionTag: null,
        isEditTag: false,
        editingTag: null,
        showArchive: false,
        complaintHistoryTag: null,
        assignPerformerTag: null,
    };
    request = new Request2({
        requestConfigs,
    });

    componentDidMount(): void {
        this.getCurrentTagsData();
    }

    shouldComponentUpdate(nextProps: Readonly<ICarCurrentTagsProps>,
        nextState: Readonly<ICarCurrentTagsState>, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    componentDidUpdate(prevProps: Readonly<ICarCurrentTagsProps>): void {
        if (this.props.carId !== prevProps.carId
            || JSON.stringify(this.props.allTagsHash) !== JSON.stringify(prevProps.allTagsHash)
            || JSON.stringify(this.props.history) !== JSON.stringify(prevProps.history)
        ) {
            this.getCurrentTagsData();

            this.setState({ complaintHistoryTag: null });
        }
    }

    getCurrentTagsData() {
        this.setState({
            carTagsIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_CAR_TAGS, { queryParams: { car_id: this.props.carId } })
                .then(response => {
                    const carTags = response;

                    const carRepairTags = this.props.allTagsHash
                        && Object.values(this.props.allTagsHash)
                            .filter(tag => tag.type === CAR_REPAIR_TAG_NAME)
                            .map(tag => tag.name) || [];

                    carTags && carTags.records.forEach(tag => {
                        if (carRepairTags.includes(tag.tag)) {
                            if (tag.query_id) {
                                tag.isCarRepairTag = true;
                            } else {
                                tag.isRequested = true;
                            }
                        }
                    });

                    carTags && carTags.records
                    && this.setState({
                        tags: carTags.records.sort((a, b) => b.priority - a.priority),
                        propositions: carTags.propositions || [],
                        proposition_users: response.proposition_users
                            && Array.isArray(response.proposition_users)
                            && response.proposition_users.reduce((_p: any, _c: any) => {
                                if (!_p.hasOwnProperty(_c.id)) {
                                    _p[_c.id] = _c;
                                }

                                return _p;
                            }, {})
                            || {},
                        carTagsIsLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        carTagsIsLoading: false,
                        error,
                    });
                });
        });

    }

    openRepairInfo(currentRepairTagId: string) {
        this.setState({ currentRepairTagId }, () => {
            this.request.exec(REQUESTS.GET_MAJOR_SERVICE_TAG_INFO, { queryParams: { id: currentRepairTagId } })
                .then(response => {
                    this.setState({ currentRepairTag: response[currentRepairTagId] });
                });
        });
    }

    closeRepairInfo() {
        this.setState({ currentRepairTagId: '' });
    }

    openRejectConfirm() {
        this.setState({ isRejectConfirmOpen: true });
    }

    closeRejectConfirm() {
        this.setState({ isRejectConfirmOpen: false, rejectError: null });
    }

    rejectServiceRequest() {
        this.setState({ isWorking: true }, () => {
            this.request.exec(REQUESTS.REJECT_MAJOR_SERVICE_TAG, { queryParams: { id: this.state.currentRepairTagId } })
                .then(() => {
                    this.setState({ isWorking: false, currentRepairTagId: '', isRejectConfirmOpen: false });
                    this.props.carId && this.getCurrentTagsData();
                })
                .catch((rejectError) => {
                    this.setState({ rejectError, isWorking: false });
                });
        });
    }

    approve(el: any) {
        this.action({
            id: el.proposition_id,
            question: `Подтвердить предложение ${el.tag}?`,
            action: REQUESTS.CAR_TAG_CONFIRM,
        });
    }

    reject(el: any) {
        this.action({
            id: el.proposition_id,
            question: `Отклонить предложение ${el.tag}?`,
            action: REQUESTS.CAR_TAG_REJECT,
        });
    }

    action(props: { question: string; action: string; id: string }) {
        this.setState({
            question: props.question,
            isWorking: false,
            confirmIsOpen: true,
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(props.action, {
                        body: {
                            proposition_ids: [props.id],
                        },
                    })
                        .then(() => {
                            this.setState({
                                isWorking: false,
                                confirmIsOpen: false,
                            }, () => {
                                this.getCurrentTagsData();
                            });
                        })
                        .catch((confirmError) => {
                            this.setState({
                                confirmError,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    onClose() {
        this.setState({
            confirmIsOpen: false,
            isWorking: false,
            addEvolution: false,
            isEditTag: false,
            editingTag: null,
        });
    }

    updateTO(tag: any) {
        const importantCarTags = this.getImportantCarTags();
        const isNeedToCheckTO = importantCarTags?.[tag.tag]?.check_to;
        const car_id = this.props.carId;

        if (!isNeedToCheckTO) {
            return;
        }

        this.setState({
            question: `Машина прошла ТО?`,
            confirmIsOpen: true,
            accept: async () => {
                let carInfo;
                try {
                    carInfo = await this.request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id } });
                } catch (confirmError) {
                    this.setState({ confirmError, isWorking: false });
                }

                const mileage = CarInfoHandler.getMileage.call(carInfo);
                const vin = CarInfoHandler.getVin.call(carInfo);
                if (vin === EMPTY_DATA) {
                    this.setState({ confirmError: new Error('отсутствует  VIN'), isWorking: false });
                }

                let tagDetails;
                try {
                    tagDetails = await this.request.exec(REQUESTS.GET_CAR_TAGS_HISTORY,
                        { queryParams: { tag_id: tag.tag_id } });
                } catch (confirmError) {
                    this.setState({ confirmError, isWorking: false });
                }

                const start_date = tagDetails?.records?.[0]?.timestamp;
                const ready_date = Math.floor(new Date().getTime() / ONE_SECOND);

                start_date && ready_date && vin !== EMPTY_DATA && mileage >= 0
                && this.setState({ isWorking: true }, () => {
                    this.request.exec(REQUESTS.UPSERT_MAINTENANCE, {
                        body:
                            { vin, mileage, start_date, ready_date },
                    })
                        .then(() => {
                            this.onClose();
                        })
                        .catch((confirmError) => {
                            this.setState({ confirmError, isWorking: false });
                        });
                });
            },
        });
    }

    removeTag(tag: any) {
        this.setState({
            question: `Закрыть тег ${tag.display_name}(${tag.tag})?`,
            confirmIsOpen: true,
            accept: () => {
                this.setState({ isWorking: true }, () => {
                    this.request.exec(REQUESTS.DELETE_CAR_TAG, { queryParams: { tag_id: tag.tag_id } })
                        .then(() => {
                            this.getCurrentTagsData();
                            this.onClose();
                            this.updateTO(tag);
                        })
                        .catch((confirmError) => {
                            this.setState({ confirmError, isWorking: false });
                        });
                });
            },
        });
    }

    editTag(tag: any) {
        this.setState({
            isEditTag: true,
            editingTag: tag,
        });
    }

    finishPerform(tag: any, type: string) {
        this.setState({
            question: `${(type === TagActions.RESET ? 'Сбросить тег ' : type === TagActions.DONE ? 'Выполнить тег ' : '')} `
                + `${tag.display_name}(${tag.tag})?`,
            confirmIsOpen: true,
            accept: () => {
                this.setState({ isWorking: true }, () => {
                    const params = type === TagActions.RESET
                        ? { drop_tag_ids: tag.tag_id }
                        : type === TagActions.DONE ? { tag_id: tag.tag_id } : {};

                    this.request.exec(REQUESTS.FINISH_PERFORM, {
                        body: params,
                        headers: { UserIdDelegation: tag.performer },
                    })
                        .then(() => {
                            this.getCurrentTagsData();
                            this.onClose();
                        })
                        .catch((confirmError) => {
                            this.setState({ confirmError, isWorking: false });
                        });
                });
            },
        });
    }

    evolve(tag: any) {
        this.setState({
            addEvolution: true,
            evolutionTag: tag,
        });
    }

    closeEvolve() {
        this.setState({ addEvolution: false }, () => {
            this.getCurrentTagsData();
        });
    }

    showArchive(showArchive) {
        this.setState({ showArchive });
    }

    getImportantCarTags() {
        return this.props.importantCarTags?.reduce((_p, _c) => {
            if (!_p.hasOwnProperty(_c.tag)) {
                _p[_c.tag] = _c;
            }

            return _p;
        }, {}) || {};
    }

    openComplaintHistory(complaintHistoryTag) {
        this.setState({ complaintHistoryTag });
    }

    assignPerformer(assignPerformerTag) {
        this.setState({ assignPerformerTag });
    }

    render() {
        const adminUser: Dict<any> = this.props.AdminUser || {};
        const evolutions = adminUser.permissions
        && adminUser.permissions.evolutions
            ? adminUser.permissions.evolutions
            : {};

        const importantCarTags = this.getImportantCarTags();

        const filteredImportantTags = this.state?.tags?.filter(item => importantCarTags?.[item.tag]?.rise_up
            //Backward Compatibility. need to remove after release 3.38.40
            || (importantCarTags?.[item.tag] && !importantCarTags?.[item.tag]?.hasOwnProperty('rise_up')));

        return <>
            {
                this.state.propositions.length
                && <>
                    <Collapse key={'propositions'}
                              title={
                                  [`Предложения для тегов `,
                                      <span key={'propositions-title'}>{this.state.carTagsIsLoading
                                          ? <Spin size={'s'}/> : `(${this.state.propositions.length})`}</span>]
                              }
                              collapsed>
                        <table className={styleTable.table}>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th/>
                                    <th>Тег</th>
                                    <th>Автор/Согласователь</th>
                                    <th/>
                                    <th/>
                                </tr>
                            </thead>
                            <tbody>
                                {
                                    this.state.propositions && this.state.propositions.map((el: any, index) => {
                                        return <tr key={index}>
                                            <td>{++index}</td>
                                            <td>
                                                <strong>
                                                    {el.hasOwnProperty('confirmations_count')
                                                && el.confirmations_count.toString()}
                                                /
                                                    {el.confirmations_need}
                                                </strong>
                                            </td>
                                            <td>
                                                {
                                                    this.props.allTagsHash
                                                && this.props.allTagsHash[el.tag]
                                                && <>
                                                    <div>{this.props.allTagsHash[el.tag].display_name}</div>
                                                    <div>{this.props.allTagsHash[el.tag].comment}</div>
                                                </>
                                                || el.tag
                                                }
                                            </td>
                                            <td>
                                                <div>
                                                    <strong>Автор: </strong>
                                                    {
                                                        this.state.proposition_users[el.proposition_author]
                                                    && this.state.proposition_users[el.proposition_author].username
                                                    && this.state.proposition_users[el.proposition_author].id
                                                            ? <Link target={'_blank'}
                                                                    href={`#/clients/${this.state.proposition_users[el.proposition_author].id}/info`}>
                                                                {this.state
                                                                    .proposition_users[el.proposition_author]
                                                                    .username}
                                                            </Link>
                                                            : EMPTY_DATA
                                                    }
                                                </div>
                                                <div>
                                                    <strong>Согласны: </strong>
                                                    {
                                                        el && el.confirmations
                                                    && Array.isArray(el.confirmations)
                                                    && el.confirmations.map((confirmation: any) => {
                                                        const username = this.state.proposition_users
                                                            && this.state
                                                                .proposition_users[confirmation.user_id]
                                                            && this.state
                                                                .proposition_users[confirmation.user_id]
                                                                .username
                                                            || EMPTY_DATA;

                                                        return <span key={confirmation.user_id}
                                                                     className={'inline-date'}>
                                                            <Link target={'_blank'}
                                                                  href={`#/clients/${confirmation.user_id}/info`}>
                                                                {username}
                                                            </Link>
                                                            :
                                                            <FormatDate withSecond
                                                                        value={
                                                                            confirmation.history_instant * ONE_SECOND
                                                                        }/>
                                                            {confirmation.comment}
                                                        </span>;
                                                    })
                                                    }
                                                </div>
                                            </td>
                                            <td>
                                                <Link onClick={this.approve.bind(this, el)}>
                                                Подтвердить
                                                </Link>
                                            </td>
                                            <td>
                                                <Link onClick={this.reject.bind(this, el)}>
                                                Отклонить
                                                </Link>
                                            </td>
                                        </tr>;
                                    })
                                }
                            </tbody>
                        </table>
                    </Collapse>
                </>
                || ''
            }
            {filteredImportantTags.length &&
            <Collapse key={'important-tags'}
                      title={
                          [`Важные теги`,
                              <span key={'important-tags-title'}>
                                  {this.state.carTagsIsLoading
                                      ? <Spin size={'s'}/>
                                      : ` (${filteredImportantTags.length})`}
                              </span>]
                      }>
                <CarCurrentTagsTable importantTagOptions={importantCarTags}
                                     carId={this.props.carId}
                                     tags={filteredImportantTags}
                                     evolutions={evolutions}
                                     editTag={this.editTag.bind(this)}
                                     evolve={this.evolve.bind(this)}
                                     removeTag={this.removeTag.bind(this)}
                                     finishPerform={this.finishPerform.bind(this)}
                                     history={this.props.history}
                                     iface_tag_implementations={this.props.iface_tag_implementations}
                                     openComplainHistory={this.openComplaintHistory.bind(this)}
                                     openRepairInfo={this.openRepairInfo.bind(this)}
                                     allTagsHash={this.props.allTagsHash}
                                     assignPerformer={this.assignPerformer.bind(this)}/>
            </Collapse> || null}
            <Collapse key={'current-tags'}
                      title={[<span key={'current_tags_title_key'}>Текущие теги. Всего: </span>,
                          <span key={'current_tags_count_key'}>{this.state.carTagsIsLoading
                              ? <Spin size={'s'}/>
                              : `${this.state?.tags?.length}`}</span>]}
                      collapsed={false}>
                <div>
                    <Link onClick={this.getCurrentTagsData.bind(this)}>
                        Обновить список тегов
                    </Link>
                </div>

                <CarCurrentTagsTable importantTagOptions={importantCarTags}
                                     carId={this.props.carId}
                                     tags={this.state?.tags}
                                     evolutions={evolutions}
                                     editTag={this.editTag.bind(this)}
                                     evolve={this.evolve.bind(this)}
                                     removeTag={this.removeTag.bind(this)}
                                     finishPerform={this.finishPerform.bind(this)}
                                     history={this.props.history}
                                     iface_tag_implementations={this.props.iface_tag_implementations}
                                     openRepairInfo={this.openRepairInfo.bind(this)}
                                     openComplainHistory={this.openComplaintHistory.bind(this)}
                                     allTagsHash={this.props.allTagsHash}
                                     assignPerformer={this.assignPerformer.bind(this)}/>

            </Collapse>
            <Button onClick={this.showArchive.bind(this, true)}>Архив тегов</Button>
            {
                this.state.showArchive
                && <TagsDeepHistory2 car_id={this.props.carId}
                                     onClose={this.showArchive.bind(this, false)}/>
            }
            <Switch>
                <Route path="/:typeID/:itemID/:itemTab/tag-detail/:tagId?"
                       render={(props) => {
                           const request = this.request.exec.bind(this.request, REQUESTS.GET_TAG_DETAILS,
                               { queryParams: { tag_id: props.match.params.tagId } });

                           return <MoreTagInfoModal header={'Подробная информация по тегу'}
                                                    request={request}
                                                    {...props}/>;
                       }}/>
                <Route path="/:typeID/:itemID/:itemTab/tag-history-details/:eventId?"
                       render={(props) => {
                           const request = this.request.exec.bind(this.request, REQUESTS.GET_CAR_TAGS_HISTORY_DETAIL,
                               { queryParams: { event_id: props.match.params.eventId } });

                           return <MoreTagInfoModal header={'Подробная информация по тегу'}
                                                    request={request}
                                                    {...props}/>;
                       }}/>
                <Route path="/:typeID/:itemID/:itemTab/car-telematic-status/:eventId?/:statusId?"
                       component={TelematicStatus}/>
            </Switch>
            {this.state.confirmIsOpen &&
            <Confirm accept={this.state.accept.bind(this)}
                     onClose={this.onClose.bind(this)}
                     isWorking={this.state.isWorking}
                     error={this.state.confirmError}
                     question={this.state.question}/>
            }
            {this.state.currentRepairTagId
            && <RepairTagInfoModal currentRepairTag={this.state.currentRepairTag}
                                   closeRepairInfo={this.closeRepairInfo.bind(this)}
                                   rejectRequest={this.openRejectConfirm.bind(this)}/>}
            {
                this.state.isRejectConfirmOpen
                && <Confirm question="Удалить заявку?"
                            error={this.state.rejectError}
                            onClose={this.closeRejectConfirm.bind(this)}
                            accept={this.rejectServiceRequest.bind(this)}
                            isWorking={this.state.isWorking}/>
            }
            {
                this.state.addEvolution && <TagModal operation={Operations.EVOLUTION}
                                                     evolutionTag={this.state.evolutionTag}
                                                     onClose={this.closeEvolve.bind(this)}/>
            }
            {
                this.state.isEditTag && <TagModal objectId={{ type: OBJECT_TYPES.CAR, id: this.props.carId ?? null }}
                                                  operation={Operations.EDIT}
                                                  isEdit={true}
                                                  initialData={this.state.editingTag ? this.state.editingTag : {}}
                                                  onClose={this.onClose.bind(this)}/>
            }
            {
                this.state.complaintHistoryTag
                    ? <ComplaintHistoryModal onClose={this.openComplaintHistory.bind(this)}
                                             tag={this.state.complaintHistoryTag}/>
                    : null
            }
            {
                this.state.assignPerformerTag
                    ? <AssignPerformerModal onClose={this.assignPerformer.bind(this, null)}
                                            tag={this.state.assignPerformerTag}/>
                    : null
            }
        </>;
    }
}
