import React from 'react';

import { Dict } from '../../../../../types';
import { EMPTY_DATA, LONGTERM_MARK, ONE_SECOND } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import FormatDate from '../../../../ui/FormatDate';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import * as styleTable from '../../../../ui/Table/index.css';
import { normalizeHash2 } from '../../../../utils/utils';
import { controlType } from '../../../FormConstructor/types';
import { COMPLAINT_TAG, SERVICE_TAG } from '../../constants';
import * as style from '../index.css';
import { TagActions } from '../types';

interface ICarCurrentTagsTableProps {
    tags: any[];
    allTagsHash: Dict<any>;
    history: any;
    iface_tag_implementations: Dict<any>;
    carId: string;
    removeTag: () => void;
    finishPerform: () => void;
    evolve: () => void;
    editTag: () => void;
    openRepairInfo: () => void;
    evolutions: Dict<any>;
    importantTagOptions: Dict<{ tag: string; check_to: boolean; rise_up: boolean }>;
    openComplainHistory: (item) => void;
    assignPerformer: (item) => void;
}

const FORMAT_DATA_TYPES = {
    TIMESTAMP: 'timestamp',
    ARRAY_TYPES: 'array_types',
};

export class CarCurrentTagsTable extends React.Component<ICarCurrentTagsTableProps> {
    getAdditionalProperty(name: string) {
        const COMMON_PARAMS = ['comment', 'priority', 'tag'];

        const _tag = this.props.allTagsHash[name] || {};
        const type = _tag?.type;
        const iface = this.props.iface_tag_implementations[type];
        const schema = iface
            ? Object.assign({}, iface)
            : {};

        schema && COMMON_PARAMS.forEach((item: string) => {
            delete schema[item];
        });

        return schema
            && Object.entries(schema)
                // don't user params start with double underscore '__'
                .filter((_el: any[]) => !~_el[0].indexOf('__'))
            || [];
    }

    formatData(props: { schema: Dict<any>; data: any }) {

        let obj: React.ReactElement | null = null;
        const visual = props.schema && props.schema.visual;
        const type = props.schema && props.schema.type;

        if (visual === FORMAT_DATA_TYPES.TIMESTAMP) {
            obj = <FormatDate value={props.data * ONE_SECOND}/>;
        } else {
            obj = props.data;
        }

        if (type === FORMAT_DATA_TYPES.ARRAY_TYPES) {
            const keys = Object.keys(props.schema.array_type);
            obj = props.data.map((_el, _i) => {
                return <div key={_i}>{
                    keys.map((__el: string, __i: number) => {
                        return _el[__el]
                            ? <span key={_i + '-' + __i}>{_el[__el]}{__i < keys.length - 1 ? ':' : ''} </span>
                            : null;
                    })
                }</div>;
            });
        }

        if ([controlType.structure].includes(type)) {
            obj = null;
        }

        return obj;
    }

    render() {
        return this.props?.tags.length && <table className={styleTable.table}>
            <tbody>
                <tr>
                    <th>#</th>
                    <th/>
                    <th>Тег</th>
                    <th colSpan={2}/>
                    <th>Комментарий</th>
                    <th/>
                </tr>
                {
                    this.props.tags.map((item, index) => {
                        const additionalProperty = this.getAdditionalProperty(item.tag) || [];
                        const closeLabel = `Закрыть${this.props.importantTagOptions?.[item.tag]?.check_to ? '+TO' : ''}`;
                        const type = this.props.allTagsHash[item.tag]?.type;

                        return <tr key={index}>
                            <td>{++index}</td>
                            <td>
                                {item.priority || EMPTY_DATA}
                            </td>
                            <td>
                                {item.display_name || item.tag}
                                {item.tag.includes(LONGTERM_MARK) && item.user_id // need to use right list of tags
                                    ?
                                    <span> <Link href={`#/clients/${item.user_id}/info-old`}
                                                 target={'_blank'}>
                                                клиент
                                    </Link></span> : ''}
                                <div className={style.tag_not_display_name}>
                                    {item.display_name !== item.tag && item.tag || ''}
                                </div>
                                {
                                    item.performer
                                && <div>Исполнитель:&nbsp;
                                    <Link href={`#/clients/${item.performer}/tags-history`}
                                          target={'_blank'}>
                                        {UserInfoHandler.getPrintNameWithoutPn.call(item?.performer_details)}
                                    </Link>
                                </div>
                                }
                                {
                                    this.props.history?.length && <hr/> || null
                                }
                                {
                                    this.props.history?.[item.tag_id]
                                        ?.sort((a, b) => b.timestamp - a.timestamp)
                                        .map((el, index) => {
                                            return <div key={index} className={style.additional_history}>
                                                {index + 1}) {el.action}: <FormatDate withSecond
                                                                                      value={
                                                                                          el.timestamp * ONE_SECOND
                                                                                      }/>
                                            : <Link href={`#/clients/${UserInfoHandler.getId.call(el.user_data_full)}/info`}
                                                    target={'_blank'}>
                                                    {UserInfoHandler.getPrintNameWithoutPn.call(el.user_data_full)}
                                                </Link>
                                            </div>;
                                        })
                                }
                            </td>
                            <td>
                                <div className={style.tag_id}>
                                    <Link href={
                                        normalizeHash2(
                                            {
                                                carId: this.props.carId,
                                                extraPath: `tag-detail/${item.tag_id}`,
                                                href: location.href,
                                            })
                                    }> JSON </Link>
                                </div>
                            </td>
                            <td>
                                {item.isCarRepairTag
                            && <span className={style.repair}
                                     onClick={this.props.openRepairInfo.bind(this, item.tag_id)}>⚙️</span>}
                                {item.isRequested
                            && <span className={style.requested}>⚙️</span>}
                            </td>
                            <td>
                                <div className={style.additional_params}>
                                    {item.comment && additionalProperty.length &&
                                <span>comment: </span> || ''}
                                    <p className={style.comment}>{item.comment}</p>
                                </div>
                                {
                                    additionalProperty && additionalProperty.map((property: any[], index) => {
                                        return property[0] && item[property[0]] !== undefined
                                        && <div className={style.additional_params} key={index}>
                                            <span>{property[0]}: </span>{
                                                this.formatData({
                                                    schema: property[1],
                                                    data: item[property[0]],
                                                })}
                                        </div>;
                                    })
                                }
                            </td>
                            <td>
                                <div>
                                    {
                                        item.tag === COMPLAINT_TAG
                                            ? <Link onClick={this.props.openComplainHistory.bind(this, item)}>
                                            История
                                            </Link>
                                            : null
                                    }
                                </div>
                                <div>
                                    {
                                        !item.performer
                                            ? <Link onClick={this.props.removeTag.bind(this, item)}>
                                                {closeLabel}
                                            </Link>
                                            : <Link onClick={this.props.finishPerform
                                                .bind(this, item, TagActions.RESET)}>
                                            Сбросить
                                            </Link>
                                    }
                                </div>
                                <div>
                                    {
                                        item.performer &&
                                    <Link onClick={this.props.finishPerform.bind(this, item, TagActions.DONE)}>
                                        Выполнить
                                    </Link>
                                    }
                                </div>
                                <div>
                                    {
                                        this.props.evolutions[item.tag]
                                            ? <Link onClick={this.props.evolve.bind(this, item)}>
                                            Эволюция
                                            </Link>
                                            : null
                                    }
                                </div>
                                <div>
                                    {
                                        type === SERVICE_TAG
                                            ? <Link onClick={this.props.assignPerformer.bind(this, item)}>
                                            Назначить
                                            </Link>
                                            : null
                                    }
                                </div>
                            </td>
                            <td>
                                <Link onClick={this.props.editTag.bind(this, item)}>
                                Редактирование
                                </Link>
                            </td>
                        </tr>;
                    })
                }
            </tbody>
        </table> || <NoInformation/>;
    }
}
