import moment from 'moment';
import * as React from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { ISearchControls, WithSearchControls } from '../../../../../decorators/SearchControls';
import { Dict, LSSettingItems } from '../../../../../types';
import Select, { IOptionInfo, WARNING_LABEL } from '../../../../ui/Select';
import LS from '../../../../utils/localStorage/localStorage';
import { Request2 } from '../../../../utils/request';
import Spin from '../../../Spin';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../../request';
import * as style from '../index.css';
import { SnapshotPhoto } from '../SnapshotPhoto';
import { ITag } from '../types';
import TagsHistoryItem from './TagsHistoryItem/index';

interface ICarTagsHistoryProps extends ISearchControls {
    carId: string;
    allTagsHash: {};
    onLoadedHistory?: (response) => void;
}

interface ICarTagsHistoryState {
    [key: string]: any;

    carTagsHistoryIsLoading: boolean;
    tagsHistory: ITag[];
    doNotFilterTagsType: boolean;
    tagsTypeOptions: IOptionInfo[];
    performersOptions: IOptionInfo[];
    selectedTags: string[];
    selectedPerformer: string[];
    showPhoto: boolean;
    photoEventId: string;
    tagPerformers: Dict<any[]> | null;
}

const TWO_DAYS = 1;

@WithSearchControls({ since: moment(new Date()).subtract(TWO_DAYS, 'd'), forceUrlParams: true })
export class CarTagsHistory extends React.Component<ICarTagsHistoryProps, ICarTagsHistoryState> {
    state: ICarTagsHistoryState = {
        carTagsHistoryIsLoading: false,
        tagsHistory: [],
        doNotFilterTagsType: false,
        tagsTypeOptions: [],
        performersOptions: [],
        selectedTags: [],
        selectedPerformer: [],
        showPhoto: false,
        photoEventId: '',
        tagPerformers: null,
    };
    request = new Request2({
        requestConfigs,
    });
    ls = new LS();

    componentDidMount(): void {
        this.getData();
        this.setState({
            doNotFilterTagsType: this.ls.get(LSSettingItems._da_doNotFilterTagsType)
                && this.ls.get(LSSettingItems._da_doNotFilterTagsType) === '1',
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    componentDidUpdate(prevProps: Readonly<ICarTagsHistoryProps>): void {
        if (this.props.carId !== prevProps.carId
            || this.props.since !== prevProps.since
            || this.props.until !== prevProps.until) {
            this.getData();
        }
    }

    showPhoto(eventId: string) {
        this.setState({
            showPhoto: true,
            photoEventId: eventId,
        });
    }

    hidePhoto() {
        this.setState({
            showPhoto: false,
            photoEventId: '',
        });
    }

    onSelectHandle(type: string, value: string) {
        this.setState({
            [type]: value,
        }, () => {
            if (type === 'selectedTags') {
                this.ls.set(LSSettingItems.defaultSelectedFilterExcept, JSON.stringify(value));
            }
        });
    }

    getData() {
        const id = this.props.carId;

        this.setState({ carTagsHistoryIsLoading: true }, () => {
            this.request.abort();
            this.request.exec(REQUESTS.GET_CAR_TAGS_HISTORY, {
                queryParams: {
                    car_id: id,
                    since: this.props.since,
                    until: this.props.until,
                },
            })
                .then(response => {

                    const uniqueTags: Dict<string> = {};
                    const uniquePerformer = new Set();

                    const tagPerformers: Dict<any[]> = response.records
                        && response.records
                            .reduce((result: Dict<any>, tagHistoryItem: Dict<any>) => {

                                if (!uniqueTags.hasOwnProperty(tagHistoryItem.tag_name)) {
                                    uniqueTags[tagHistoryItem.tag_name] = tagHistoryItem.tag_display_name;
                                }

                                tagHistoryItem.user_data_full && tagHistoryItem.user_data_full.username
                                && uniquePerformer.add(tagHistoryItem.user_data_full.username);

                                const { tag_id, user_id, timestamp, action } = tagHistoryItem;
                                let userData = { user_id, timestamp, action };
                                if (tagHistoryItem.user_data_full) {
                                    const {
                                        first_name = '',
                                        last_name = '',
                                        username = '',
                                    } = tagHistoryItem.user_data_full;
                                    userData = Object.assign(userData, { first_name, last_name, username });
                                }

                                if (!result[tag_id]) {
                                    result[tag_id] = [userData];
                                } else {
                                    result[tag_id].push(userData);
                                }

                                return result;
                            }, {});

                    let tagsTypeOptions = Object.entries(uniqueTags)
                        .map((item: [string, string]) => ({
                            text: item[1],
                            value: item[0],
                            description: item[0],
                        }));

                    let selectedTags = [];
                    if (this.ls.get(LSSettingItems.defaultSelectedFilterExcept) && !this.state.doNotFilterTagsType) {
                        selectedTags =
                            JSON.parse(this.ls.get(LSSettingItems.defaultSelectedFilterExcept) || '[]') || [];
                        selectedTags.forEach((item: string) => {
                            const match = tagsTypeOptions.filter((i: IOptionInfo) => i.value == item);

                            const text = this.props.allTagsHash
                                && this.props.allTagsHash[item]
                                && (this.props.allTagsHash[item].display_name
                                    || this.props.allTagsHash[item].name)
                                || item;

                            !match.length && (tagsTypeOptions = [
                                {
                                    text: `${WARNING_LABEL} ${text}`,
                                    value: item,
                                    description: ' - прошлый выбор',
                                }
                                , ...tagsTypeOptions]);
                        });
                    }

                    const performersOptions: IOptionInfo[] = Array.from(uniquePerformer).map((item: string) => {
                        return { text: item, value: item };
                    });

                    this.props?.onLoadedHistory?.(response.records);

                    this.setState({
                        carTagsHistoryIsLoading: false,
                        tagsHistory: response.records,
                        tagsTypeOptions,
                        performersOptions,
                        selectedTags,
                        tagPerformers,
                    });
                });
        });
    }

    render() {
        const tagsHistory = this.state.tagsHistory && this.state.tagsHistory.length
            ? this.state.tagsHistory.filter((item) => {
                if (this.state.doNotFilterTagsType) {
                    return true;
                }

                return (this.state.selectedTags.includes(item.tag_name)
                            && (item.user_data_full
                                && this.state.selectedPerformer.includes(item.user_data_full.username)))
                        || (!this.state.selectedTags.length
                            && (item.user_data_full
                                && this.state.selectedPerformer.includes(item.user_data_full.username)))
                        || (!this.state.selectedPerformer.length
                            && this.state.selectedTags.includes(item.tag_name))
                        || (!this.state.selectedTags.length && !this.state.selectedPerformer.length);

            })?.sort((a, b) => b.timestamp - a.timestamp)
            : [];

        return !this.state.carTagsHistoryIsLoading
            ? <div>
                {
                    !this.state.doNotFilterTagsType && <div className={style.filter_container}>
                        <Select options={this.state.tagsTypeOptions}
                                placeholder={'Фильтр по тегу'}
                                containerClassName={style.filter}
                                onSelect={this.onSelectHandle.bind(this, 'selectedTags')}
                                multiSelect={true}
                                initialValues={this.state.selectedTags}/>
                        <Select options={this.state.performersOptions}
                                placeholder={'фильтр по исполнителю'}
                                containerClassName={style.filter}
                                onSelect={this.onSelectHandle.bind(this, 'selectedPerformer')}
                                multiSelect={true}/>
                    </div>
                }
                {tagsHistory && tagsHistory.length
                    ? <div className={style.tags_history_list}>
                        <div className={style.content}>
                            <div className={style.header}>
                                <span className={style.cell}>#</span>
                                <span className={style.cell}>Дата</span>
                                <span className={style.cell}>Действие</span>
                                <span className={style.cell}/>
                                <span className={style.cell}>Тег</span>
                                <span className={style.cell}>Пользователь</span>
                                <span className={style.cell}>Автор</span>
                                <span className={style.cell}/>
                            </div>
                            <VirtualList width={'100%'}
                                         height={900}
                                         itemCount={tagsHistory.length}
                                         overscanCount={300}
                                         itemSize={45}
                                         renderItem={({ index, style }) => {
                                             return <TagsHistoryItem _style={style}
                                                                     index={index}
                                                                     item={tagsHistory[index]}
                                                                     showPhoto={this.showPhoto.bind(this)}
                                                                     key={index}/>;
                                         }}/>
                        </div>
                    </div>
                    : <h3>История не найдена</h3>
                }
                {
                    this.state.showPhoto
                        ? <SnapshotPhoto eventId={this.state.photoEventId} onClose={this.hidePhoto.bind(this)}/>
                        : null
                }
            </div>

            : <Spin/>;
    }
}
