import moment from 'moment';
import * as React from 'react';
import ReactJson from 'react-json-view';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import DatePicker from '../../../../ui/DatePicker';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { NoInformation } from '../../../../ui/NoInformation';
import styleTable from '../../../../ui/Table/index.css';
import { Tabs } from '../../../../ui/Tabs';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../../request';
import { ITelematicsSensor } from '../component';
import style from './index.css';

const request = new Request2({ requestConfigs });
const DAYS = 5;

interface ICLickHouseModalProps {
    sensor: ITelematicsSensor | null;
    onClose: () => void;
    carId: string;
}

enum TabsTitle {
    TABLE = 'Table',
    RAW = 'RAW',
}

export const ClickHouseModal = (props: ICLickHouseModalProps) => {
    const { sensor, onClose, carId } = props;

    const [since, setSince] = React.useState(moment().subtract(DAYS, 'days'));
    const [until, setUntil] = React.useState(moment());
    const [selectedTab, setSelectedTab] = React.useState<TabsTitle>(TabsTitle.TABLE);
    const tabs = [
        { name: TabsTitle.TABLE.toUpperCase(), link: TabsTitle.TABLE },
        { name: TabsTitle.RAW.toUpperCase(), link: TabsTitle.RAW },
    ];

    const options = {
        requestName: REQUESTS.GET_TELEMATICS_HISTORY,
        requestOptions: {
            queryParams: {
                car_id: carId,
                source: 'ch',
                sensor_id: sensor?.id,
                sensor_subid: sensor?.subid,
                since: Math.floor(+since / ONE_SECOND),
                until: Math.floor(+until / ONE_SECOND),
            },
        },
    };

    const [isLoading, response, error, getClickHouseHistory] = useRequestHandler<any>(request, options);

    React.useEffect(() => {
        getClickHouseHistory();
    }, [since, until]);

    return (
        <Window title={`История сенсора ${sensor?.name}`}
                onClose={onClose}>
            {
                isLoading
                    ? <Spin/>
                    : <div className={style.history}>
                        <div>
                            <DatePicker value={since}
                                        onChange={(value) => setSince(value)}
                                        placeholder={'От'}
                                        className={style.datePicker}/>
                            <DatePicker value={until}
                                        onChange={(value) => setUntil(value)}
                                        placeholder={'До'}
                                        className={style.datePicker}/>
                        </div>
                        <Tabs selectTab={(tab: TabsTitle) => setSelectedTab(tab)}
                              tabs={tabs}
                              currentTab={selectedTab}/>
                        {
                            error
                                ? <SimpleError error={error}/>
                                : selectedTab === TabsTitle.TABLE
                                    ? <table className={styleTable.table}>
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Время</th>
                                                <th>Значение</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            {response?.sensors
                                                ? response.sensors[0]?.values.map((historyItem, index) => {
                                                    return <tr key={historyItem.timestamp + index}>
                                                        <td>{index + 1}</td>
                                                        <td>{historyItem.timestamp &&
                                                            <FormatDate withSecond
                                                                        value={historyItem.timestamp * ONE_SECOND}/>}
                                                        </td>
                                                        <td>{historyItem.value ?? EMPTY_DATA}</td>
                                                    </tr>;
                                                })
                                                : <tr>
                                                    <td><NoInformation/></td>
                                                </tr>
                                            }
                                        </tbody>
                                    </table>
                                    : <ReactJson src={response}
                                                 displayObjectSize={false}
                                                 displayDataTypes={false}
                                                 enableClipboard={false}
                                                 collapsed={4}/>
                        }
                    </div>
            }
        </Window>
    );
};
