import * as React from 'react';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import * as styleTable from '../../../ui/Table/index.css';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CarContext } from '../context';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { ClickHouseModal } from './ClickHouseModal';
import * as style from './index.css';

const sensorItemTooltipId = 'sensorItemTooltipId';

export interface ITelematicsSensor {
    id: number;
    name: string;
    since: number;
    updated: number;
    value: any;
    subid?: number;
}

interface ICarTelematicsProps {
    carId: string;
}

interface ICarTelematicsState {
    sensors: ITelematicsSensor[];
    sensorsTranslation: Dict<any>;
    isLoading: boolean;
    loadingError: null;
    editingSensor: ITelematicsSensor | null;
    isEditWindowOpen: boolean;
    isUpdating: boolean;
    editingSensorValue: string;
    setValueError: Error | null;
    isSensorHistoryOpen: boolean;
    isClickHouseHistoryOpen: boolean;
    sensorHistory: ITelematicsSensor | null;
    carInfo: Dict<any>;
    filter: string;
}

export default class CarTelematics extends React.Component<ICarTelematicsProps, ICarTelematicsState> {
    state: ICarTelematicsState = {
        sensors: [],
        sensorsTranslation: {},
        isLoading: false,
        isEditWindowOpen: false,
        loadingError: null,
        isUpdating: false,
        editingSensor: null,
        editingSensorValue: '',
        setValueError: null,
        isSensorHistoryOpen: false,
        isClickHouseHistoryOpen: false,
        sensorHistory: null,
        carInfo: {},
        filter: '',
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ICarTelematicsProps>): void {
        if (this.props.carId !== prevProps.carId) {
            this.getData();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    getData() {
        this.setState({
            isLoading: true,
        }, () => {

            this.request.exec(REQUESTS.GET_TELEMATICS_STATE, { queryParams: { car_id: this.props.carId } })
                .then(response => {
                    this.setState({
                        sensors: response.sensors || [],
                        isLoading: false,
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError });
                });

            this.request.exec(REQUESTS.GET_TELEMATICS_JSON)
                .then(response => {
                    let data = response?.settings[0]?.setting_value;
                    if (isValidJSONString(data)) {
                        data = JSON.parse(data).items;
                        const sensorsTranslationMap = {};
                        data.map(el => {
                            const elId = this.generateSensorId(el);
                            sensorsTranslationMap[elId] = el;
                        });

                        this.setState({
                            sensorsTranslation: sensorsTranslationMap,
                        });
                    }
                });
        });
    }

    openSensorEditor(editingSensor: ITelematicsSensor) {
        this.setState({ isEditWindowOpen: true, editingSensor, editingSensorValue: editingSensor.value });
    }

    closeSensorEditor() {
        this.setState({
            isEditWindowOpen: false,
            editingSensor: null,
            editingSensorValue: '',
            setValueError: null,
            isUpdating: false,
        });
    }

    onChange(editingSensorValue: string) {
        this.setState({ editingSensorValue });
    }

    setNewValue() {
        this.setState({ isUpdating: true }, () => {
            const body = {
                car_id: this.props.carId,
                command: 'SET_PARAM',
                id: this.state.editingSensor && this.state.editingSensor.id,
                value: this.state.editingSensorValue && +this.state.editingSensorValue,
                ...this.state.editingSensor?.subid && { subid: this.state.editingSensor.subid },
            };

            this.request.exec(REQUESTS.CAR_CONTROL, { body }).then(() => {
                this.closeSensorEditor();
                this.getData();
            }).catch((setValueError) => {
                this.setState({ setValueError, isUpdating: false });
            });
        });
    }

    openSensorHistory(sensorHistory: ITelematicsSensor) {
        this.setState({ isSensorHistoryOpen: true, sensorHistory });
    }

    openClickHouseHistory(sensorHistory: ITelematicsSensor) {
        this.setState({ isClickHouseHistoryOpen: true, sensorHistory });
    }

    closeClickHouseHistory() {
        this.setState({ isClickHouseHistoryOpen: false, sensorHistory: null });
    }

    closeSensorHistory() {
        this.setState({ isSensorHistoryOpen: false, sensorHistory: null });
    }

    onFilterChange(filter) {
        this.setState({ filter }, () => {
            ReactTooltip.rebuild();
        });
    }

    generateTooltipTemplate(data) {
        let template = '';
        for (const key in data) {
            template += `<div><span>${[key]}: </span><span>${data[key]}</span></div>`;
        }

        return template;
    }

    generateSensorId(sensor) {
        return `${sensor.id}${sensor.subid ? `${sensor.subid}` : ''}`;
    }

    render() {
        const { sensorsTranslation } = this.state;

        if (this.state.loadingError) {
            return <SimpleError error={this.state.loadingError} data={{ label: 'Ошибка при загрузке данных' }}/>;
        }

        return <div>
            <Input placeholder={'фильтр'}
                   onChange={this.onFilterChange.bind(this)}
                   value={this.state.filter}
                   className={style.inputFilter}/>
            {this.state.isLoading
                ? <Spin/>
                : this.state?.sensors?.length
                    ? <table className={styleTable.table}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>id/sub id</th>
                                <th>Сенсор</th>
                                <th>Значение</th>
                                <th>Начало</th>
                                <th>Обновлено</th>
                                <th/>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <th><ReactTooltip id={sensorItemTooltipId} type={'info'} html/></th>
                            </tr>
                            {this.state.sensors
                                .filter(sensor => {
                                    if (!this.state.filter) {
                                        return true;
                                    }

                                    return sensor.id.toString().includes(this.state.filter) ||
                                    sensor.subid?.toString().includes(this.state.filter) ||
                                    sensor.value?.toString().includes(this.state.filter) ||
                                    sensor.name.toString().toUpperCase().includes(this.state.filter.toUpperCase());
                                })
                                .map((sensor, index) => {
                                    const sensorTranslationItem = sensorsTranslation[this.generateSensorId(sensor)];

                                    return <tr key={`${sensor.id}_${sensor.name}`}>
                                        <td>{index + 1}</td>
                                        <td>{sensor.id}{sensor.subid ? `/${sensor.subid}` : ''}</td>
                                        <td>
                                            <Link onClick={this.openSensorHistory.bind(this, sensor)}
                                                  className={style.sessionHistoryLink}>
                                                {sensor.name}
                                            </Link>
                                        (<Link onClick={this.openClickHouseHistory.bind(this, sensor)}>
                                        история ch</Link>)
                                            {
                                                sensorTranslationItem
                                                    ? <span className={style.sessionHistoryLink}
                                                            data-for={sensorItemTooltipId}
                                                            data-tip={
                                                                this.generateTooltipTemplate(sensorTranslationItem)
                                                            }>
                                                        {sensorTranslationItem.name}
                                                    </span>
                                                    : null
                                            }
                                        </td>
                                        <td className={style.sensor_val}>{sensor.value}</td>
                                        <td>
                                            <FormatDate value={sensor.since * ONE_SECOND} withSecond/>
                                        </td>
                                        <td>
                                            <FormatDate value={sensor.updated * ONE_SECOND} withSecond/>
                                        </td>
                                        <td>
                                            <Link onClick={this.openSensorEditor.bind(this, sensor)}>
                                                Редактирование
                                            </Link>
                                        </td>
                                    </tr>;
                                })}
                        </tbody>
                    </table>
                    : <h3>Нет данных</h3>}
            {this.state.isEditWindowOpen
                ? <Window error={this.state.setValueError}
                          onClose={this.closeSensorEditor.bind(this)}
                          title={`Изменить значение сенсора ${this.state.editingSensor && this.state.editingSensor.name}`}>
                    <Input placeholder={'Новое значение'}
                           value={this.state.editingSensorValue}
                           onChange={this.onChange.bind(this)}/>
                    <div className={style.button_container}>
                        <Button onClick={this.closeSensorEditor.bind(this)}
                                basic
                                colorType={ButtonTypes.negative}>Отмена</Button>
                        <Button isLoading={this.state.isUpdating}
                                onClick={this.setNewValue.bind(this)}>Подтвердить</Button>
                    </div>
                </Window>
                : null}
            {this.state.isSensorHistoryOpen && this.context.imei
                ? <SensorHistoryModal imei={this.context.imei}
                                      sensor={this.state.sensorHistory}
                                      onClose={this.closeSensorHistory.bind(this)}/>
                : null}

            {this.state.isClickHouseHistoryOpen && this.props.carId
                ? <ClickHouseModal sensor={this.state.sensorHistory}
                                   onClose={this.closeClickHouseHistory.bind(this)}
                                   carId={this.props.carId}/>
                : null}
        </div>;
    }
}

CarTelematics.contextType = CarContext;

interface ISensorHistoryItem {
    ts: number;
    value: any;
    id: number;
    timestamp?: number;
}

interface ISensorHistoryModalProps {
    imei: string;
    sensor: ITelematicsSensor | null;
    onClose: () => void;
}

interface ISensorHistoryModalState {
    loadingError: Error | null;
    isLoading: boolean;
    history: ISensorHistoryItem[];
}

export class SensorHistoryModal extends React.Component<ISensorHistoryModalProps, ISensorHistoryModalState> {
    state: ISensorHistoryModalState = {
        loadingError: null,
        isLoading: false,
        history: [],
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ISensorHistoryModalProps>): void {
        if (this.props.imei !== prevProps.imei) {
            this.getData();
        }
    }

    getData() {
        this.props.imei && this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_TELEMATICS_HISTORY, {
                queryParams: {
                    imei: this.props.imei,
                },
            })
                .then(response => {
                    const item = response[this.props.imei].find(el => {
                        return this.props.sensor?.subid
                            ? (el?.subid === this.props.sensor?.subid) && el.id === this.props?.sensor?.id
                            : el.id === this.props?.sensor?.id;
                    });
                    this.setState({
                        history: item?.timeline?.reverse() || [],
                        isLoading: false,
                    });
                })
                .catch(loadingError => {
                    this.setState({ loadingError });
                });
        });
    }

    render() {
        return <Window title={`История сенсора ${this.props.sensor && this.props.sensor.name}`}
                       onClose={this.props.onClose}>
            {this.state.loadingError
                ? <SimpleError error={this.state.loadingError}/>
                : this.state.isLoading
                    ? <Spin/>
                    : this.state.history && this.state.history.length
                        ? <div className={style.history}>
                            <table className={styleTable.table}>
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Время</th>
                                        <th>Значение</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {this.state.history.map((historyItem, index) => {
                                        return <tr key={historyItem.ts}>
                                            <td>{index + 1}</td>
                                            <td>{historyItem.ts &&
                                        <FormatDate withSecond value={historyItem.ts * ONE_SECOND}/>}</td>
                                            <td>{historyItem.value !== undefined ? historyItem.value : EMPTY_DATA}</td>
                                        </tr>;
                                    })}
                                </tbody>
                            </table>
                        </div>
                        : <h3>Нет данных</h3>}
        </Window>;
    }
}
