import React from 'react';

import { Dict } from '../../../../types';
import { TIMERS } from '../../../constants';
import { Button, ButtonTypes, CancelButton } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import { Link } from '../../../ui/Link';
import { CarStatuses } from '../../../ui/Status';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { deepCopy, generateuuid4 } from '../../../utils/utils';
import { FormConstructor } from '../../FormConstructor';
import style from './index.css';
import { CAR_REQUESTS, REQUESTS } from './request';
import { schema } from './schema';

interface ISendCarToChatProps {
    carInfo: Dict<any>;
    blockRules: Dict<any>;
}

interface ISendCarToChatState {
    modalIsOpen: boolean;
}

const CHAT_TAG_AUTO = 'support_chat_where_auto';

class SendCarToChat extends React.Component<ISendCarToChatProps, ISendCarToChatState> {
    state: ISendCarToChatState = {
        modalIsOpen: false,
    };

    shouldComponentUpdate(
        nextProps: Readonly<ISendCarToChatProps>,
        nextState: Readonly<any>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(nextProps?.carInfo, this.props?.carInfo) || !isObjectEqual(this.state, nextState);
    }

    openModal(modalIsOpen) {
        this.setState({
            modalIsOpen,
        });
    }

    render() {
        const { carInfo, blockRules } = this.props;
        const { modalIsOpen } = this.state;

        return blockRules?.OutgoingChat && carInfo.status === CarStatuses.AVALIABLE ? <>
            <Button colorType={ButtonTypes.positive}
                    onClick={this.openModal.bind(this, true)}>Отправить авто в чат</Button>
            {
                modalIsOpen
                    ? <SendCarToChatModal carInfo={carInfo}
                                          onClose={this.openModal.bind(this, false)}/>
                    : null
            }
        </> : null;
    }
}

export { SendCarToChat };

interface ISendCarToChatModalProps {
    onClose: () => void;
    carInfo: Dict<any>;
}

interface ISendCarToChatModalState {
    isLoading: boolean;
    data: Dict<any>;
    isValid: boolean;
    error: Error | null;
    topic_link: string | null;
    queryForResend: null | Dict<any>;
}

const COUNT_LIMIT = 5;

class SendCarToChatModal extends React.Component<ISendCarToChatModalProps, ISendCarToChatModalState> {
    state: ISendCarToChatModalState = {
        isLoading: false,
        error: null,
        data: {},
        isValid: false,
        topic_link: null,
        queryForResend: null,
    };
    timeout;
    sendMessageCount = 0;

    request = new Request2({ requestConfigs: CAR_REQUESTS });

    componentWillUnmount() {
        this.request.abort();
        clearTimeout(this.timeout);
    }

    //1 - create chat
    //2 - perform chat
    //3 - send message
    //4 - drop perform

    retrySendMessage(props) {
        clearTimeout(this.timeout);
        ++this.sendMessageCount;

        this.timeout = setTimeout(() => {
            this.sendMessage(props);
        }, TIMERS.UPDATE_SELECTED);
    }

    manualResend() {
        const queryForResend = this.state.queryForResend && deepCopy(this.state.queryForResend);
        this.sendMessageCount = 0;
        this.setState({
            error: null,
            isLoading: true,
            queryForResend: null,
        }, () => {
            this.sendMessage(queryForResend);
        });
    }

    createChat() {
        const topic_link = `support.${generateuuid4()}`;
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.CREATE_CHAT, {
                body: {
                    initial_node: 'intro_common',
                    tag: CHAT_TAG_AUTO,
                    object_id: this.state.data.object_id,
                    priority: 0,
                    topic_link,
                },
            })
                .then((response) => {
                    const tag_id = response?.tagged_objects?.[0].tag_id?.[0];
                    tag_id && this.performChat(tag_id, topic_link);
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });

        });
    }

    performChat(tag_id, chat_id) {
        this.request.exec(REQUESTS.PERFORM_CHAT, {
            body: {
                tag_id,
            },
        })
            .then(() => {
                this.sendMessage({
                    user_id: this.state.data.object_id,
                    chat_id,
                    message: this.state.data.comment,
                    tag_id,
                });
            })
            .catch(error => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    defer(tag_id) {
        this.request.exec(REQUESTS.DEFER, {
            queryParams: {
                tag_id,
            },
        })
            .then(() => {
                this.setState({
                    isLoading: false,
                }, () => {
                    this.props.onClose();
                });
            })
            .catch(error => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    sendMessage({ user_id, chat_id, message, tag_id }) {
        this.request.exec(REQUESTS.SEND_MESSAGE, {
            queryParams: {
                user_id,
                chat_id,
            },
            body: { message, 'message_type': 'plaintext' },
        })
            .then(() => {
                this.defer(tag_id);
            })
            .catch(error => {
                if (this.sendMessageCount < COUNT_LIMIT) {
                    this.retrySendMessage({ user_id, chat_id, message, tag_id });
                } else {
                    this.setState({
                        error,
                        queryForResend: { user_id, chat_id, message, tag_id },
                        isLoading: false,
                    });
                }
            });
    }

    onFormChange(data, isValid) {
        this.setState({
            data, isValid,
        });
    }

    render() {
        const { onClose, carInfo } = this.props;
        const isLoading = this.state.isLoading;
        const carDeeplink = `yandexdrive://cars/${carInfo?.number}`;

        return <Window title={'Создать чат и отправить в него ссылку на автомобиль'}
                       error={this.state.error}
                       onClose={onClose.bind(this)}>
            <FormConstructor className={style.overflow}
                             schema={schema}
                             initialData={{
                                 comment: `Нужную машину вы найдёте по ссылке `
                        + `[url=${carDeeplink}]${carDeeplink}[/url]`,
                             }}
                             onChange={this.onFormChange.bind(this)}/>
            <div>
                {
                    this.state.queryForResend &&
                    <h4> Не получилось отправить сообщение, пожалуйста кликните <Link onClick={this.manualResend
                        .bind(this)}>сюда</Link> для повторной попытки!!!
                    </h4>
                }
            </div>
            <div className={coreStyle.button_container}>
                <CancelButton isLoading={isLoading}
                              onClick={this.props.onClose.bind(this)}/>
                <Button onClick={this.createChat.bind(this)}
                        isLoading={isLoading}
                        disabled={!this.state.isValid}>Отправить</Button>
            </div>
        </Window>;
    }
}
