import * as React from 'react';

import { Dict } from '../../../types';
import { OLD_SESSION_TAB_ID, SESSION_TAB_ID } from '../../constants';
import { Cross } from '../../ui/Cross';
import * as coreStyle from '../../ui/index.css';
import { hasWebphonePanel } from '../../utils/hasWebphonePanel';
import { Request2 } from '../../utils/request';
import { ytLogs } from '../../utils/sendLogs';
import { buttonNameDetails } from '../../utils/sendLogs/eventTypes/buttonDetails';
import { EVENT_TYPES } from '../../utils/sendLogs/eventTypes/eventTypes';
import { TabView } from '../../utils/TabView';
import { setDocumentTitle } from '../../utils/utils';
import { IStore } from '../App/store';
import { VisitedLog } from '../Content/db';
import Map2 from '../Map2';
import { SimpleError } from '../SimpleError';
import CarCardHeader from './CarCardHeader';
import CarCardSidebar from './CarCardSidebar';
import { CarContext } from './context';
import * as style from './index.css';
import { CAR_REQUESTS as requestConfigs, REQUESTS } from './request';

interface ICarCardProps extends IStore {
    carId: string;
    location: any;
}

interface ICarCardState {
}

export default class CarCard extends React.Component<ICarCardProps, ICarCardState> {
    request = new Request2({ requestConfigs });

    closeCarCard() {
        const queries = new URLSearchParams(this.props.location
            && this.props.location.search);

        if (queries && queries.get('back')) {
            location.hash = queries.get('back')!;
        } else {
            location.hash = '#/cars/';
        }
    }

    render() {
        return <>
            <div className={style.map_container}>
                {
                    this.props.AdminUser?.blockRules?.HideLargeMap && (typeof window.orientation !== 'undefined')
                        ? <div>Карта временно недоступна на мобильных устройствах :(</div>
                        : <Map2 location={this.props.location}
                                carCardOpen={!!this.props.carId}/>
                }
            </div>
            {this.props.carId
                ? <CarCardModal {...this.props}
                                carId={this.props.carId}
                                title={'Автомобиль'}
                                onClose={this.closeCarCard.bind(this)}/>
                : null}
        </>;
    }
}

const CAR_INFO_TIMEOUT = 3000;
const RETRY_LIMIT = 5;

interface IState {
    isLoading: boolean;
    carInfo: Dict<any>;
    loadingError: Error | null;
    passwordTelematics: string | null;
}

export class CarCardModal extends React.Component<any, IState> {
    log = ytLogs.getInstance();
    state: IState = {
        isLoading: false,
        carInfo: {},
        loadingError: null,
        passwordTelematics: null,
    };
    request = new Request2({ requestConfigs });
    visitedLog = new VisitedLog();
    timer: any;
    limit = 0;

    showPassword() {
        if (!this.state.passwordTelematics) {
            const car_id: string = this.state.carInfo?.id || this.props.carId;

            this.log.send({
                data: {
                    event_type: EVENT_TYPES.CLICK_BUTTON,
                    button_name: buttonNameDetails.SHOW_TELEMATICS_PASSWORD,
                    car_id,
                },
            });

            this.request.exec(REQUESTS.GET_PASSWORD, { queryParams: { car_id } })
                .then(response => {
                    this.setState({
                        passwordTelematics: response.password,
                    });
                })
                .catch(error => {
                    this.setState({
                        passwordTelematics: null,
                    });
                });
        } else {
            this.setState({
                passwordTelematics: null,
            });
        }
    }

    update() {
        clearTimeout(this.timer);
        this.timer = setTimeout(() => {
            this.getData({ firstTime: false });
        }, CAR_INFO_TIMEOUT);
    }

    componentDidMount() {
        this.getData({ firstTime: true });
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<any>): void {
        if (prevProps.carId !== this.props.carId) {
            this.getData({ firstTime: true });
        }
    }

    componentWillUnmount() {
        clearTimeout(this.timer);
        this.request.abort();
    }

    getData(opt: { firstTime: boolean }) {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_CAR_INFO, { queryParams: { car_id: this.props.carId } })
                .then(response => {
                    setDocumentTitle(response.number);
                    this.limit = 0;
                    this.setState({
                        carInfo: response,
                        isLoading: false,
                        loadingError: null,
                    });

                    if (opt.firstTime) {
                        this.visitedLog.saveLog(response, this.props);
                        this.setState({
                            passwordTelematics: null,
                        });
                    }

                    this.update();
                })
                .catch(loadingError => {
                    if (this.limit < RETRY_LIMIT) {
                        this.getData({ firstTime: false });
                        this.limit++;
                    } else {
                        this.setState({ loadingError });
                    }
                });
        });
    }

    render() {
        const props: any = this.props;
        const isCurrentTab = this.props?.match?.params?.tabId === OLD_SESSION_TAB_ID;
        const isCurrentSession = this.props?.match?.params?.tabId === SESSION_TAB_ID;
        const isSession2Tab = isCurrentSession || (isCurrentTab && this.props?.AdminUser?.blockRules?.NewSessionList);
        const hasWebphone = hasWebphonePanel(location.href, this.props?.AdminUser?.blockRules);
        const carInfo = Object.keys(this.state.carInfo)?.length && this.state.carInfo || { id: this.props.carId };

        return <div className={coreStyle.card_modal}>
            <div className={coreStyle.background} onClick={this.props.onClose.bind(this)}/>
            <div className={`${coreStyle.card_container} ${!hasWebphone ? coreStyle.hasNoWebphone : ''}`}>
                <div className={coreStyle.card}>
                    <div className={coreStyle.header}>
                        <CarCardHeader isLoading={this.state.isLoading} carInfo={this.state.carInfo}/>
                        {this.state.loadingError
                            ? <SimpleError error={this.state.loadingError}
                                           data={{ label: 'Ошибка при загрузке данных' }}/>
                            : null}
                    </div>
                    <div className={coreStyle.tabs}>
                        <TabView withoutTitle {...props}/>
                    </div>
                    <div className={`${coreStyle.sidebar} ${style.sidebar_wrapper}`}>
                        <CarCardSidebar carInfo={carInfo}
                                        passwordTelematics={this.state.passwordTelematics}
                                        showPassword={this.showPassword.bind(this)}
                                        getData={this.getData.bind(this)}/>
                    </div>
                    <div className={`${coreStyle.content} ${isSession2Tab ? coreStyle.session_content_car_fix : ''}`}>
                        <CarContext.Provider value={this.state.carInfo}>
                            {this.props.children}
                        </CarContext.Provider>
                    </div>
                </div>
                <Cross onClick={this.props.onClose} className={coreStyle.close_icon}/>
            </div>
        </div>;
    }
}
