import * as React from 'react';
import { useState } from 'react';
import VirtualList from 'react-tiny-virtual-list';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { useWindowResize } from '../../../hooks/useWindowResize';
import CarInfo from '../../../models/car';
import Checkbox from '../../../ui/Checkbox';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import { CarStatus } from '../../../ui/Status';
import { formatNumberToStringWithSpaces, truncateString } from '../../../utils/utils';
import CarNumber from '../../CarNumber';
import Spin from '../../Spin';
import { ICarFiltered } from '../component';
import { EDGE_MARGIN, FONT_SIZE } from '../constants';
import * as style from './index.css';

const CARS_MARGIN_EM = 16;
const CAR_ITEM_SIZE = 45;
const TABLE_MARGIN = 18;

interface ICarsListProps {
    dataIsLoading: boolean;
    cars: ICarFiltered[];
    showComment: boolean;
    setFilteredCars: (id: string, checked: boolean, all: boolean) => void;
    carsWithCheckboxLength: number;
    firstNCars: number;
    setFirstNCars: (num) => void;
}

const carCurrentTooltipId = 'car-tooltip';

export const CarsList = React.memo((props: ICarsListProps) => {
    const {
        dataIsLoading = false,
        cars = [],
        showComment,
        setFilteredCars,
        carsWithCheckboxLength,
        firstNCars,
        setFirstNCars,
    } = props;
    const windowSize = useWindowResize();
    const tableHeight = windowSize.height - EDGE_MARGIN - FONT_SIZE * CARS_MARGIN_EM - TABLE_MARGIN;
    const [globalCheckbox, setGlobalCheckbox] = useState(true);

    const onChange = (checked) => {
        setGlobalCheckbox(checked);
        setFilteredCars('', checked, true);
    };

    React.useEffect(() => {
        setGlobalCheckbox(true);
    }, [cars.length]);

    return <div className={style.cars}>
        {
            dataIsLoading
                ? <Spin/>
                : cars.length
                    ? <div>
                        <div className={style.cars_list_length}>
                            <Input value={firstNCars}
                                   onChange={setFirstNCars}
                                   placeholder={'Выбор первых N авто'}
                                   type={'number'}
                                   className={style.input}/>
                            <span>Всего автомобилей: </span>
                            <span className={style.count}>
                                {formatNumberToStringWithSpaces(carsWithCheckboxLength)}
                            </span>
                        </div>

                        <div className={`${style.header}` + ` ${showComment ? style.extraRow : ''}`}>
                            <span className={style.cell}>
                                <Checkbox plain
                                          checked={globalCheckbox}
                                          onChange={onChange}/>
                            </span>
                            <span className={style.cell}>#</span>
                            <span className={style.cell}>Номер</span>
                            <span className={style.cell}>Модель</span>
                            <span className={style.cell}>Статус</span>
                            <span className={style.cell}>Топливо/Пробег</span>
                            <span className={style.cell}/>
                            <span className={style.cell}/>
                            {
                                showComment
                                    ? <span className={style.cell}>{'Комментарии'}</span>
                                    : null
                            }
                        </div>
                        <ReactTooltip id={carCurrentTooltipId}
                                      html/>
                        <VirtualList height={tableHeight}
                                     itemCount={cars.length}
                                     itemSize={CAR_ITEM_SIZE}
                                     renderItem={({ index, style }) => {
                                         const item = cars[index];

                                         return <CarListItem key={index}
                                                             showComment={showComment}
                                                             index={index}
                                                             _style={style}
                                                             item={item}
                                                             setFilteredCars={setFilteredCars}
                                                             setGlobalCheckbox={setGlobalCheckbox}/>;
                                     }}/>
                    </div>
                    : <NoInformation/>

        }
    </div>;
});

export type carIds = {
    [key: string]: boolean;
}

interface ICarListItemProps {
    showComment: boolean;
    item: ICarFiltered;
    index: number;
    _style: Dict<any>;
    setFilteredCars:(id: string, checked: boolean, all: boolean) => void;
    setGlobalCheckbox: (checked) => void;
}

const MAX_CHARS = 38;

const CarListItem = React.memo((props: ICarListItemProps) => {
    const { item: car, index, _style, showComment, setFilteredCars, setGlobalCheckbox } = props;
    const mileage = car?.telematics?.mileage?.toFixed(1) ?? null;

    const onChange = (checked) => {
        setFilteredCars(car.id, checked, false);
        if(!checked) {
            setGlobalCheckbox(checked);
        }
    };

    return <div style={_style} className={`${style.row}` + ` ${showComment ? style.extraRow : ''}`}>
        <span className={style.cell}>
            <Checkbox plain
                      checked={car.checked}
                      onChange={onChange}/>
        </span>
        <span className={style.cell}>{index + 1}</span>
        <span className={style.cell} title={car.number}>
            {
                car.number
                    ?
                    <CarNumber carInfo={car as unknown as typeof CarInfo} externalTooltipId={carCurrentTooltipId}/>
                    : EMPTY_DATA
            }
        </span>
        <span className={style.cell} title={car.model_id}>
            <Link target={'_blank'} href={`#/cars/${car.id}/info`}>
                {car.model_id}
            </Link>
        </span>
        <span className={style.cell}>
            <CarStatus text={car.status}/>
        </span>
        <span className={style.cell}>
            {car?.telematics?.fuel_level || mileage
                ? <div className={style.telematics}>
                    <span className={style.fuel_level}>
                        {car.telematics?.fuel_level ?? EMPTY_DATA} <span>%</span>
                    </span>
                    <span>{mileage ? mileage + ' км' : EMPTY_DATA}</span>
                </div>
                : null
            }
        </span>
        <span className={style.cell}>
            {car.usage}
        </span>
        <span className={style.cell}>
            <Link target={'_blank'}
                  href={`#/tracks?car_id=${car.id}&status=`}>
                Треки
            </Link>
        </span>
        {
            showComment
                ? <span title={car?.tags?.[0]?.comment ?? ''}
                        className={style.cell}>
                    {truncateString((car?.tags?.map(el => el.comment)
                        .filter(el => el)
                        .join('; ') || ''), MAX_CHARS)}
                </span>
                : null
        }
    </div>;
});
