import * as React from 'react';

import { Dict } from '../../../../types';
import CarInfo from '../../../models/car';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { deepCopy } from '../../../utils/utils';
import CarNumber from '../../CarNumber';
import QueryScheduler, { COUNT_FIRST, IProgressData } from '../../QueryScheduler/QueryScheduler';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CARS_FILTER_REQUESTS, REQUESTS } from '../request';
import * as style from './index.css';

interface IGroupAttachModalProps {
    onClose: () => void;
    data: {
        tag: string;
        comment: string;
        priority: number;
        isProposition: boolean;
        cars: string[];
        [x: string]: any;
    }[];
    openOnFinish?: boolean;
}

interface IGroupAttachModalState {
    hasDuplicates: Dict<any>;
    hasNoDuplicates: Dict<any>;
    progress: string;
    errors: any[];
    tagsPriority: Dict<number>;
    deprecatedTags: string[];
    isLoading: boolean;
}

const defaultTag = (item, defaultPriority) => ({
    ...item,
    comment: item?.comment ?? '',
    priority: item?.priority ?? defaultPriority,
    cars: [],
});

export class GroupAttachModal extends React.Component<IGroupAttachModalProps, IGroupAttachModalState> {
    state: IGroupAttachModalState = {
        hasDuplicates: {},
        hasNoDuplicates: {},
        progress: '',
        errors: [],
        tagsPriority: {},
        deprecatedTags: [],
        isLoading: false,
    };

    request = new Request2({ requestConfigs: CARS_FILTER_REQUESTS });

    componentDidMount(){
        this.request.exec(REQUESTS.GET_TAGS)
            .then((response) => {
                const tagsPriority = response?.records?.reduce((acc, tag) => {
                    acc[tag.name] = +tag.default_priority;

                    return acc;
                }, {});
                const deprecatedTags = response?.deprecated.reduce((acc, tag) => {
                    acc.push(tag.name);

                    return acc;
                }, []);

                this.setState({
                    tagsPriority,
                    deprecatedTags,
                });
            });
        this.checkForDuplicates();
    }

    checkForDuplicates() {
        const { data } = this.props;
        const tag_queue: Set<Promise<any>> = new Set();
        const car_id_queue: Set<string> = new Set();

        data?.forEach((tag) => {
            tag.cars.forEach((car_id) => {
                tag_queue.add(this.request.exec.bind(this.request, REQUESTS.GET_CAR_TAG_LIST, {
                    queryParams: {
                        car_id: car_id,
                    },
                }));
                car_id_queue.add(car_id);
            });
        });

        const qs_tags = new QueryScheduler({
            queue: Array.from(tag_queue),
            limit: COUNT_FIRST,
            onProgress: this.onProgress.bind(this),
            onSuccess: this.onSuccessGetTags.bind(this, Array.from(car_id_queue)),
        });

        qs_tags.run();

        this.setState({
            isLoading: true,
        });
    }

    onProgress() {}

    getTagnameError(tag: string): string | null {
        if (this.state.deprecatedTags.length && this.state.deprecatedTags.includes(tag)) {
            return 'удален, его больше нельзя навесить';
        }

        const tags = Object.keys(this.state.tagsPriority);
        if (tags.length && !tags.includes(tag)) {
            return 'не существует, проверьте написание';
        }

        return null;
    }

    onSuccessGetTags(car_id_queue: string[], response: IProgressData) {
        const { data } = this.props;
        const { tagsPriority } = this.state;
        let allCurrentTags = response.success;
        const hasNoDuplicates: any = {};
        const hasDuplicates: any = {};
        const car_queue: Set<string> = new Set();

        allCurrentTags = car_id_queue.map((car_id) => {
            return allCurrentTags.find((car) => {
                return car?.records?.[0]?.object_id === car_id;
            });
        });

        if (response.failed.length) {
            this.setState({
                errors: response.failed,
            });
            response.failed.forEach((failed) => {
                allCurrentTags.splice(failed.data, 0, {});
            });
        }

        data.forEach((item) => {
            if (!this.getTagnameError(item.tag)) {
                item.cars.forEach((car_id) => {
                    const cars_tags = allCurrentTags
                        ?.find((currentTagData) => currentTagData?.records?.[0]?.object_id === car_id) ?? null;
                    if (cars_tags?.records?.find((tag) => tag.tag === item.tag)) {
                        if (!hasDuplicates[item.tag]) {
                            hasDuplicates[item.tag] = defaultTag(item, tagsPriority[item.tag]);
                        }

                        hasDuplicates[item.tag].cars.push(car_id);
                        car_queue.add(car_id);
                    } else {
                        if (!hasNoDuplicates[item.tag]) {
                            hasNoDuplicates[item.tag] = defaultTag(item, tagsPriority[item.tag]);
                        }

                        hasNoDuplicates[item.tag].cars.push(car_id);
                    }
                });
            } else {
                this.setState((prev) => ({
                    errors: [
                        ...prev.errors,
                        {
                            error: new Error(`Тег ${item.tag} ${this.getTagnameError(item.tag)}`),
                        },
                    ],
                }));
            }
        });

        this.setState({
            hasNoDuplicates,
        }, () => {
            if (!Object.keys(hasDuplicates).length) {
                this.attachTag(this.state.hasNoDuplicates);
            } else {
                const qs_car_info = new QueryScheduler({
                    queue: Array.from(car_queue)
                        .map((car_id) => this.request.exec.bind(this.request, REQUESTS.GET_CAR_INFO, {
                            queryParams: {
                                car_id,
                            },
                        },
                        )),
                    limit: COUNT_FIRST,
                    onProgress: this.onProgress.bind(this),
                    onSuccess: this.onSuccessGetCar.bind(this, hasDuplicates),
                });

                qs_car_info.run();
            }

            this.setState({
                isLoading: false,
            });
        });
    }

    onSuccessGetCar(hasDuplicates, response: IProgressData) {
        Object.keys(hasDuplicates).forEach((key) => {
            hasDuplicates[key].cars = response?.success
                ?.filter((car) => hasDuplicates[key].cars.includes(car.id))
                ?.concat(response.failed.map((error) => hasDuplicates[key].cars[error.data]));
        });

        this.setState(({
            hasDuplicates,
        }));
    }

    attachTag(data, showWarning?) {
        if (Object.keys(data).length) {
            const queue: Promise<any>[] = [];
            Object.keys(data).forEach((tag) => {
                data[tag].cars.forEach((car) => {
                    const carId = car?.id ?? car.toString();
                    const handler = data[tag].isProposition ? REQUESTS.CAR_TAG_PROPOSE : REQUESTS.ATTACH_CAR_TAG;
                    const body = deepCopy(data[tag]);
                    delete body.isProposition;
                    delete body.cars;
                    body['object_id'] = carId;

                    queue.push(this.request.exec.bind(
                        this.request,
                        handler,
                        {
                            queryParams: { object_id: carId },
                            body,
                        },
                    ));
                });
            });

            const qs = new QueryScheduler({
                queue,
                limit: COUNT_FIRST,
                onProgress: this.onProgressAttach.bind(this),
                onSuccess: this.onSuccessAttach.bind(this),
            });

            qs.run();
        } else {
            showWarning && this.props.openOnFinish
                ? this.setState({
                    progress: 'Нет данных для навешивания тегов!',
                })
                : this.props.onClose();
        }
    }

    onProgressAttach(data: IProgressData) {
        const progress = data.failed.length
            ? `${data.success.length}(${data.failed.length})/${data.queue.length}`
            : `${data.success.length}/${data.queue.length}`;
        this.setState({ progress: 'Обработка автомобилей... ... ' + progress });
    }

    onSuccessAttach(data: IProgressData) {
        if (!data.failed.length && !this.state.errors.length) {
            this.props.openOnFinish
                ? this.setState({ progress: 'Завершено!' })
                : this.props.onClose();
        } else {
            this.setState((prev) => ({
                errors: [...prev.errors, ...data.failed],
            }));
        }
    }

    forceAttachTags() {
        this.attachTag(this.state.hasNoDuplicates);
        this.attachTag(this.state.hasDuplicates);
    }

    render() {
        const { hasDuplicates, progress, errors, hasNoDuplicates, isLoading } = this.state;
        const { onClose } = this.props;

        return (
            <Window className={style.group_attach}
                    onClose={onClose.bind(this)}
                    title={'Пакетное навешивание тега'}>
                {isLoading && <Spin/>}
                {progress && <p>{progress}</p>}
                {!progress.length && Object.keys(hasDuplicates).map((tag, index) => {
                    return (
                        <React.Fragment key={`dupl-${index}`}>
                            <p>
                                Тег <b>{tag}</b> уже навешен на следующие автомобили:
                            </p>
                            <table className={tblStyle.table}>
                                <tbody>
                                    {hasDuplicates[tag].cars.map((car, index) => {
                                        return (
                                            <tr key={index}>
                                                <td>{index + 1}</td>
                                                <td>
                                                    <CarNumber carInfo={car as typeof CarInfo}/>
                                                </td>
                                                <td>
                                                    <Link href={`#/cars/${car?.id}/`}>
                                                        {car?.model_id ?? car?.id}
                                                    </Link>
                                                </td>
                                            </tr>
                                        );
                                    })}
                                </tbody>
                            </table>
                        </React.Fragment>
                    );
                })}
                {!progress.length && Object.keys(hasDuplicates).length
                    ? (
                        <>
                            <p>Что делать?</p>
                            <div>
                                <Button className={style.button}
                                        colorType={ButtonTypes.negative}
                                        onClick={this.forceAttachTags.bind(this)}>Все равно навесить</Button>
                                {hasNoDuplicates
                                    && <Button className={style.button}
                                               onClick={this.attachTag.bind(this, hasNoDuplicates, true)}>
                                        Навесить только на остальные
                                    </Button>}
                                <Button className={style.button}
                                        basic
                                        onClick={onClose.bind(this)}>Отмена</Button>
                            </div>
                        </>
                    )
                    : null
                }
                {errors && errors.map((error, index) => {
                    return <SimpleError key={`error-${index}`}
                                        error={error.error}
                                        data={{ label: error?.data?.toString() || undefined }}/>;
                })}
            </Window>
        );
    }
}
