import React, { useState } from 'react';
import VirtualList from 'react-tiny-virtual-list';
import ReactTooltip from 'react-tooltip';

import { Dict } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { useWindowResize } from '../../../hooks/useWindowResize';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { WARNING_LABEL } from '../../../ui/Select';
import Spin from '../../Spin';
import { EDGE_MARGIN, FONT_SIZE } from '../constants';
import { ICarTag } from '../types';
import * as style from './index.css';

const TAGS_MARGIN_EM = 4;
const ITEM_SIZE = 90;

interface ITagsListProps {
    filterValue: string;
    dataIsLoading: boolean;
    changeFilterValue: () => void;
    carTags: ICarTag[];
    selectTag: () => void;
    onChangeSort: (func: Function) => void;
}

const sort1 = (a, b) => a?.name?.localeCompare(b?.name);
export const sort2 = (a, b) => +(b?.description?.default_priority || 0) - +(a?.description?.default_priority || 0)
    || a?.description?.display_name?.localeCompare(b?.description?.display_name)
    || a?.name?.localeCompare(b?.name);
const sort3 = (a, b) => b?.cars_count - a?.cars_count;

const handlersSort = {
    'sort1': { func: sort1, tooltip: 'Сортировка по имени тега (name)' },
    'sort2': { func: sort2, tooltip: 'Сортировка по убыванию приоритета + display_name + name' },
    'sort3': { func: sort3, tooltip: 'Сортировка по кол-ву автомобилей' },
};
const tooltipId = 'sortTooltip';

export const TagsList = React.memo((props: ITagsListProps) => {
    const { filterValue, dataIsLoading, changeFilterValue, carTags, selectTag, onChangeSort } = props;
    const windowSize = useWindowResize();
    const [active, selectSort] = useState('sort2');

    const clickHandler = (name: string) => {
        selectSort(name);
        onChangeSort(handlersSort[name].func);
    };

    return <div className={style.filter_tags}>
        <div className={style.filter_tag_input_container}>
            <Input value={filterValue}
                   className={style.filter_tag_input}
                   onChange={changeFilterValue.bind(null)}
                   disabled={dataIsLoading}
                   placeholder={'Фильтр тегов'}/>
        </div>
        <ReactTooltip id={tooltipId}/>
        <div>
            <span>Сортировка: </span>
            {
                Object.entries(handlersSort).map((el, index) => {
                    return <span className={`${style.sort_item} ${active === el[0] ? style.active : ''}`}
                                 key={index}
                                 data-tip={el[1].tooltip}
                                 data-for={tooltipId}
                                 onClick={clickHandler.bind(null, el[0])}>{el[0].substr(-1)}</span>;
                })
            }
        </div>
        <div className={`${style.tags_list_container} ${dataIsLoading ? style.loading : ''}`}>
            {
                dataIsLoading
                    ? <Spin/>
                    :
                    <VirtualList height={windowSize.height - EDGE_MARGIN - FONT_SIZE * TAGS_MARGIN_EM}
                                 itemCount={carTags.length}
                                 itemSize={ITEM_SIZE}
                                 renderItem={({ index, style }) => {
                                     const item = carTags[index];

                                     return <TagListItem key={item.name}
                                                         _style={style}
                                                         item={item}
                                                         selectTag={selectTag.bind(null, item)}/>;
                                 }
                                 }/>
            }
        </div>
    </div>;
});

interface ITagListItemProps {
    item: ICarTag;
    _style: Dict<any>;
    selectTag: () => void;
}

const TagListItem = React.memo((props: ITagListItemProps) => {
    const { item, _style, selectTag } = props;
    const { description, cars_count } = item;
    const { name, default_priority, comment, display_name } = description;

    const itemStyle = Object.assign({}, _style,
        {
            margin: `.1em .1em`,
            width: `calc(100% - .6em)`,
            height: `calc(${ITEM_SIZE}px - .2em)`,
        });

    return <div style={itemStyle} className={style.filter_item} onClick={selectTag.bind(null)}>
        <div className={style.name} title={name}>
            {name ?? WARNING_LABEL}
        </div>
        <div className={style.display_name} title={display_name}>
            {display_name ?? WARNING_LABEL}
        </div>
        <div className={style.default_priority} title={default_priority}>
            {default_priority?.toString() ?? EMPTY_DATA}
        </div>
        <div className={style.cars_count} title={cars_count.toString() ?? EMPTY_DATA}>
            {cars_count ?? EMPTY_DATA}
        </div>
        <div className={style.comment} title={comment}>
            {comment ?? EMPTY_DATA}
        </div>
        <div className={style.exclude}>
            <Link onClick={selectTag.bind(null, true)}>[-]</Link>
        </div>
    </div>;
});
