import * as React from 'react';

import { Dict, ICar, LSSettingItems } from '../../../types';
import { defaultAntitraits, defaultTraits, mileageSensor } from '../../constants';
import { CarInfoHandler } from '../../models/car';
import { Collapse } from '../../ui/Collapse';
import { Window } from '../../ui/FullModal';
import LS from '../../utils/localStorage/localStorage';
import { Request2 } from '../../utils/request';
import { deepCopy } from '../../utils/utils';
import { setCoordinates } from '../Map2/helpers/setCoordinates';
import TagModal from '../TagModal';
import { CarsList } from './CarsList/component';
import { FILTERING_TIMEOUT } from './constants';
import { FilterQuery } from './FilterQuery/component';
import { GeotagFilters } from './GeotagFilters/component';
import GroupAttachModal from './GroupAttachModal';
import GroupModal from './GroupModal';
import * as style from './index.css';
import { OfferFilters } from './OfferFilters/component';
import { CARS_FILTER_REQUESTS, REQUESTS } from './request';
import { TagComments } from './TagComments/component';
import { sort2, TagsList } from './TagsList/component';
import { CAR_FILTERS, CarFilters, FilterItem, ICarOffer, ICarTag, LogicOperator } from './types';
import { WalletFilters } from './WalletFilters/component';

declare let ymaps;

const currentOfferTrait = 'ReportCurrentOffer';
const tagCommentTrait = 'ReportTagDetails';
const geotagTrait = 'ReportLocationDetails';

export interface ICarFiltered extends ICar {
    checked: boolean;
}

interface ICarsFilterProps {
    onClose: () => void;
    downloadClick: () => void;
    toMapClick: () => void;
    clearClick: () => void;
    customPolygon: any;
}

interface ICarsFilterState {
    dataIsLoading: boolean;
    cars: ICarFiltered[];
    allTags: ICarTag[];
    tagsArray: ICarTag[];
    filterValue: string;
    sortFunction: any;
    filterQuery: string;
    selectedTag: string;
    currentLogicOperator: LogicOperator;
    error: Error | null;
    filterByPoly: boolean;
    polyCars: any[];
    groupHandlerIsOpen: boolean;
    tagModalIsOpen: boolean;
    attachData: any;
    filters: CarFilters<string[]>;
    isActive: CarFilters<boolean>;
    filteredCars: ICarFiltered[];
    offers: ICarOffer[];
    useCheckboxFilter: boolean;
    firstNCars: number;
}

export class CarsFilter extends React.Component<ICarsFilterProps, ICarsFilterState> {
    ls = new LS();
    state: ICarsFilterState = {
        dataIsLoading: false,
        cars: [],
        allTags: [],
        tagsArray: [],
        filterValue: '',
        sortFunction: null,
        selectedTag: '',
        filterQuery: this.ls.get(LSSettingItems.tags_filter) ?? '',
        currentLogicOperator: LogicOperator.AND,
        error: null,
        filterByPoly: false,
        polyCars: [],
        groupHandlerIsOpen: false,
        tagModalIsOpen: false,
        attachData: null,
        filters: {
            [CAR_FILTERS.offer]: [],
            [CAR_FILTERS.wallet]: [],
            [CAR_FILTERS.tag]: [],
            [CAR_FILTERS.geotag]: [],
        },
        isActive: {
            [CAR_FILTERS.offer]: false,
            [CAR_FILTERS.wallet]: false,
            [CAR_FILTERS.tag]: false,
            [CAR_FILTERS.geotag]: false,
        },
        filteredCars: [],
        offers: [],
        useCheckboxFilter: false,
        firstNCars: 0,
    };
    timeOutFiltering;
    request = new Request2({ requestConfigs: CARS_FILTER_REQUESTS });

    componentDidMount() {
        this.getFilteredCars(true);
    }

    componentDidUpdate(prevProps: Readonly<ICarsFilterProps>, prevState: Readonly<ICarsFilterState>): void {
        if (this.state.filterQuery !== prevState.filterQuery) {
            if (this.timeOutFiltering) {
                clearTimeout(this.timeOutFiltering);
            }

            this.timeOutFiltering = setTimeout(() => {
                this.setState({
                    error: null,
                    dataIsLoading: true,
                }, () => {
                    this.getFilteredCars();
                });
            }, FILTERING_TIMEOUT);
        }
    }

    getFilteredCars(isMount?: boolean) {
        const { filterQuery } = this.state;
        const { traits, antitraits } = this.getRequestTraits();

        this.setState({
            dataIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_CARS, {
                queryParams: {
                    sensors: mileageSensor,
                    tags_filter: encodeURIComponent(filterQuery),
                    antitraits,
                    traits,
                },
            })
                .then(response => {
                    const { cars = [], tags_array = [], offers_array } = response;

                    this.setState({
                        cars: this.addCheckboxStateToCars(cars),
                        dataIsLoading: false,
                        filteredCars: this.addCheckboxStateToCars(cars),
                        tagsArray: tags_array,
                        offers: offers_array,
                    }, () => {
                        if (isMount) {
                            this.setState({ allTags: [...tags_array] });
                        } else {
                            this.filterCars();
                        }
                    });
                })
                .catch(error => {
                    this.setState({ error, dataIsLoading: false });
                });
        });
    }

    addCheckboxStateToCars(cars) {
        return cars.map(car => {
            car.checked = true;

            return car;
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getRequestTraits(): { traits: string; antitraits: string } {
        const { isActive } = this.state;

        const addComments = isActive[CAR_FILTERS.tag] || isActive[CAR_FILTERS.wallet];
        const addGeotags = isActive[CAR_FILTERS.geotag];
        const addOffers = isActive[CAR_FILTERS.offer];

        const traits = defaultTraits
            + (addComments ? `,${tagCommentTrait}` : '')
            + (addGeotags ? `,${geotagTrait}` : '');

        const antitraits = defaultAntitraits
            + (addOffers ? '' : `,${currentOfferTrait}`);

        return { traits, antitraits };
    }

    changeFilterValue(filterValue: string) {
        this.setState({ filterValue });
    }

    getFilterTailValue(): LogicOperator {
        return this.state.filterQuery?.[this.state.filterQuery.length - 1] as LogicOperator;
    }

    filterTags() {
        const { filterValue } = this.state;
        const tail = this.getFilterTailValue();
        let tags = tail === LogicOperator.OR ? this.state.allTags : this.state.tagsArray;

        tags = filterValue
            ? tags.filter(tag => {
                const { description = {} as Dict<any> } = tag;
                const { display_name = '', comment = '', name = '' } = description;
                const lowerCaseFilterValue = filterValue.toLowerCase();

                return [display_name, comment, name].some(value => value.toLowerCase().includes(lowerCaseFilterValue));
            })
            : tags;
        tags = tags.filter(carTag => carTag.cars_count);

        return tags;
    }

    selectTag(tag: ICarTag, exclude?: boolean): void {
        const tail = this.getFilterTailValue();
        const tag_name = exclude === true ? `-${tag.name}` : tag.name;
        const isTailNotLogicOperator = tail && ![LogicOperator.OR, LogicOperator.AND].includes(tail);
        const filterQuery = `${this.state.filterQuery}`
            + `${isTailNotLogicOperator ? this.state.currentLogicOperator : ''}`
            + `${tag_name}`
            + `${this.state.currentLogicOperator}`;

        this.setState({ filterQuery });
    }

    fallLastChar(str: string) {
        return str.slice(0, str.length - 1);
    }

    onLogicOperatorChange(currentLogicOperator: LogicOperator) {
        const query = this.state.filterQuery;
        const operator: LogicOperator = query.slice(-1) as LogicOperator;
        const newData = Object.assign(
            {},
            { currentLogicOperator },
            (operator !== currentLogicOperator && operator)
                ? {
                    filterQuery:
                        ([LogicOperator.OR, LogicOperator.AND].includes(operator)
                            ? this.fallLastChar(query)
                            : query) + currentLogicOperator,
                }
                : { filterQuery: '' });
        this.setState(newData);
    }

    changeTextarea(filterQuery: string) {
        this.setState({ filterQuery: filterQuery.trim() });
    }

    check(cars: any[] = []) {
        let objects: any = null;

        ymaps.ready(() => {
            objects = ymaps.geoQuery(
                cars
                    .map((car: ICarFiltered) => new ymaps.Placemark(setCoordinates({
                        lon: car.location?.lon,
                        lat: car.location?.lat,
                    }), { car })),
            );

            const objectsInsidePoly = this.props.customPolygon
                && objects && objects?.searchInside(this.props.customPolygon);

            let polyCars: any = [];

            if (objectsInsidePoly?.getLength()) {
                objectsInsidePoly.each((item: any) => {
                    polyCars.push(item.properties.get('car'));
                });
            }

            polyCars = this.addCheckboxStateToCars(polyCars);

            this.setState(() => ({ polyCars }), () => {
                this.filterCars();
            });
        });
    }

    changeCheckbox() {
        this.setState((prev) => ({
            filterByPoly: !prev.filterByPoly,
        }), () => {
            this.check(this.state.cars);
        });
    }

    getFilteredCarsByPoly() {
        const polyCarsId: string[] = this.state?.polyCars?.map((car: ICarFiltered) => car.id);

        return !this.state.dataIsLoading && this.state?.cars
            ?.filter((car: ICarFiltered) => {
                return this.state.filterByPoly
                    ? polyCarsId?.includes(car.id)
                    : true;
            }) || [];
    }

    tagModal() {
        this.setState((prev) => ({
            tagModalIsOpen: !prev.tagModalIsOpen,
        }));
    }

    customTagModal(data) {
        this.setState({
            attachData: data,
        });
    }

    groupModal() {
        this.setState((prev) => ({
            groupHandlerIsOpen: !prev.groupHandlerIsOpen,
        }));
    }

    attachmentHandler(data, isProposition) {
        const cars = this.getCarsForRenderWithCheckboxFilter()?.map((car) => CarInfoHandler.getId.call(car));

        this.customTagModal({
            ...data,
            isProposition,
            cars: cars,
        });
        this.tagModal();
    }

    clearMapAndQuery() {
        this.props.clearClick();
        this.changeTextarea('');
    }

    onChangeSort(sortFunction) {
        this.setState({
            sortFunction,
        });
    }

    setFilter(filter: FilterItem<string[]>) {
        this.setState((prevState) => ({
            filters: {
                ...prevState.filters,
                ...filter,
            },
        }), () => this.filterCars());
    }

    setActive(filter: FilterItem<boolean>) {
        this.setState((prevState) => ({
            isActive: {
                ...prevState.isActive,
                ...filter,
            },
        }), () => {
            // update cars only if filter is active
            if (Object.values(filter)?.[0]) {
                this.getFilteredCars();
            }
        });
    }

    filterCars() {
        const { filters } = this.state;
        let filteredCars = deepCopy(this.getFilteredCarsByPoly());

        const offerOptions = filters[CAR_FILTERS.offer];
        const walletOptions = filters[CAR_FILTERS.wallet];
        const tagOptions = filters[CAR_FILTERS.tag];
        const geotagOptions = filters[CAR_FILTERS.geotag];

        if (offerOptions.length) {
            filteredCars = filteredCars.filter(car => {
                return offerOptions.includes(car.offer_group_name ?? '') || offerOptions.includes(car.offer_name ?? '');
            });
        }

        if (walletOptions.length) {
            filteredCars = filteredCars.filter(car => {
                return car.tags?.some(tag => walletOptions.includes(tag.parent_id ?? ''));
            });
        }

        if (tagOptions.length) {
            filteredCars = filteredCars.map(car => {
                car.tags = car?.tags?.filter((car) => tagOptions?.includes(car.tag));

                return car;
            });
        }

        if (geotagOptions.length) {
            filteredCars = filteredCars.filter(car => {
                return car.location?.areas?.some(el => geotagOptions.includes(el.id));
            });
        }

        this.setState({ filteredCars });
    }

    getCarsForRenderWithCheckboxFilter() {
        let cars = deepCopy(this.state.filteredCars);

        if (this.state.firstNCars > 0) {
            cars = cars.slice(0, this.state.firstNCars);
        }

        if (this.state.useCheckboxFilter) {
            cars = cars.filter(car => car.checked);
        }

        return cars;
    }

    setFilteredCars(id: string, checked: boolean, all: boolean) {
        let newFilteredCars = deepCopy(this.state.filteredCars);

        if (all) {
            newFilteredCars = newFilteredCars.map(car => {
                car.checked = checked;

                return car;
            });
        } else {
            const carToChange = newFilteredCars.find(car => car.id === id);
            carToChange.checked = checked;
        }

        this.setState({
            polyCars: newFilteredCars,
            filteredCars: newFilteredCars,
            useCheckboxFilter: true,
        });
    }

    setFirstNCars(number) {
        this.setState({
            firstNCars: number,
        });
    }

    render() {
        const {
            error,
            filterValue,
            dataIsLoading,
            filterQuery,
            currentLogicOperator,
            filterByPoly,
            tagModalIsOpen,
            attachData,
            tagsArray,
            filters,
            cars,
            offers,
        } = this.state;

        const carTags = this.filterTags()?.sort?.(this.state.sortFunction || sort2);
        let carsForRender = this.state.filteredCars;

        if (this.state.firstNCars > 0) {
            carsForRender = carsForRender.slice(0, this.state.firstNCars);
        }

        const carsWithCheckbox = this.getCarsForRenderWithCheckboxFilter();
        const showTagComment = Boolean(filters[CAR_FILTERS.tag].length);

        return <Window title={'Фильтр по машинам'}
                       onClose={this.props.onClose.bind(this)}
                       error={error}>
            {tagModalIsOpen &&
            <TagModal onClose={this.tagModal.bind(this)}
                      attachmentHandler={this.attachmentHandler.bind(this)}
                      queueLength={carsWithCheckbox?.length}/>
            }
            {attachData &&
            <GroupAttachModal onClose={this.customTagModal.bind(this, null)}
                              data={[attachData]}/>
            }
            {this.state.groupHandlerIsOpen &&
            <GroupModal cars={carsWithCheckbox}
                        isWorking={false}
                        onClose={this.groupModal.bind(this)}/>
            }
            <div className={style.car_filters}>
                <TagsList filterValue={filterValue}
                          dataIsLoading={dataIsLoading}
                          changeFilterValue={this.changeFilterValue.bind(this)}
                          onChangeSort={this.onChangeSort.bind(this)}
                          carTags={carTags}
                          selectTag={this.selectTag.bind(this)}/>
                <FilterQuery filterQuery={filterQuery}
                             onLogicOperatorChange={this.onLogicOperatorChange.bind(this)}
                             changeTextarea={this.changeTextarea.bind(this)}
                             currentLogicOperator={currentLogicOperator}
                             downloadClick={this.props.downloadClick.bind(this, carsWithCheckbox, showTagComment)}
                             clearClick={this.clearMapAndQuery.bind(this)}
                             toMapClick={this.props.toMapClick.bind(this)}
                             groupHandler={this.groupModal.bind(this)}
                             attachTag={this.tagModal.bind(this)}
                             filterByPoly={filterByPoly}
                             changeCheckBox={this.changeCheckbox.bind(this)}
                             disable={!cars.length}
                             cars={cars}/>
                <Collapse title={'Развернуть фильтры'}
                          force={true}
                          className={style.collapse}
                          collapsed={true}
                          children={
                              <span className={style.filter_wrap}>
                                  <TagComments cars={cars}
                                               filterQuery={filterQuery}
                                               setFilter={this.setFilter.bind(this)}
                                               setActive={this.setActive.bind(this)}
                                               tagsArray={tagsArray}/>

                                  <WalletFilters cars={cars}
                                                 filterQuery={filterQuery}
                                                 setFilter={this.setFilter.bind(this)}
                                                 setActive={this.setActive.bind(this)}/>

                                  <OfferFilters cars={cars}
                                                offers={offers}
                                                filterQuery={filterQuery}
                                                setActive={this.setActive.bind(this)}
                                                setFilter={this.setFilter.bind(this)}/>

                                  <GeotagFilters cars={cars}
                                                 filterQuery={filterQuery}
                                                 setActive={this.setActive.bind(this)}
                                                 setFilter={this.setFilter.bind(this)}/>
                              </span>
                          }/>
                <CarsList dataIsLoading={dataIsLoading}
                          showComment={showTagComment}
                          cars={carsForRender}
                          setFilteredCars={this.setFilteredCars.bind(this)}
                          carsWithCheckboxLength={carsWithCheckbox?.length}
                          firstNCars={this.state.firstNCars}
                          setFirstNCars={this.setFirstNCars.bind(this)}/>
            </div>
        </Window>;
    }
}
