import React from 'react';
import { RouteComponentProps } from 'react-router-dom';

import ChatIcon from '../../../svg-components/chat.component.svg';
import { CALL_ROLES_TAG, CHAT_ROLES_TAG, DUTY_ROLES_TAG, DUTY_STATE } from '../../constants';
import { BackgroundTabs } from '../../utils/BackgroundTabs';
import { CSSettingsItems, CustomSettings } from '../../utils/customSettings';
import { isObjectEqual } from '../../utils/isObjectEqual';
import LS from '../../utils/localStorage/localStorage';
import { NAVIGATION } from '../../utils/navigation';
import { ChatNotification } from '../ChatsCommon/ChatNotification';
import { _CHATS, NChats } from '../ChatsCommon/types';
import MY_ROLES_LIST from '../Core/classes/MyRoleList';
import { AbsenceModal } from '../Duty/AbsenceModal';
import DutyEnter from '../Duty/EnterModal';
import { FromBreakConfirm } from '../Duty/FromBreakConfirm';
import { FAV_LINES_TYPES, FavouritesLines } from '../FavouritesLines';
import { INotifyMapStateToProps } from './index';
import * as style from './index.css';
import CHATS_COMMAND = NChats.CHATS_COMMAND;

export const isFeedState = function (this, state) {
    return this.props.myChats?.status_tags?.length
        && this.props.myChats?.status_tags[0].tag === state
        || this.props?.status_tags?.length
        && this.props?.status_tags[0].tag === state;
};

const MORE_COUNT = 99;
const CHATS_PATH = '/' + NAVIGATION.ROUTES.CHATS.uri;

interface ChatNotifyProps extends INotifyMapStateToProps, RouteComponentProps {
    setChats: (payload: any, error: any) => void;
    setChatsLoading: (payload: { chatType: _CHATS; state: boolean }) => void;
}

export class ChatNotify extends React.Component<ChatNotifyProps, any> {
    worker = new Worker(new URL('./../../../workers/check-chats-notify.worker', import.meta.url));
    favLines = new FavouritesLines();
    chatNotification = new ChatNotification();
    cs = new CustomSettings()?.getSettings() || {};

    onSlave() {
        this.finishChecking();
    }

    onMaster() {
        const { isChatRole, isCallRole, isDutyRole } = this.getReasons();
        this.initChecking(isChatRole, isCallRole, isDutyRole);
    }

    componentDidUpdate(prevProps: Readonly<ChatNotifyProps>, prevState: Readonly<any>, snapshot?: any) {
        const ShowDisableFeedBtnRule = this.props.AdminUser?.blockRules?.ShowDisableFeedBtn;
        const prevShowDisableFeedBtnRule = prevProps.AdminUser?.blockRules?.ShowDisableFeedBtn;
        const pathName = this.props.location.pathname;
        const isFeedDisabled = this.checkForDisableFeed();

        if (ShowDisableFeedBtnRule !== prevShowDisableFeedBtnRule) {
            (!isFeedDisabled || pathName === CHATS_PATH) && this.startWorker();
        } else if (pathName !== prevProps.location.pathname) {
            if (isFeedDisabled && this.props.location.pathname === CHATS_PATH) {
                this.finishChecking();
                this.startWorker();
            } else if (isFeedDisabled && pathName !== CHATS_PATH) {
                this.finishChecking();
            }
        }
    }

    checkForDisableFeed() {
        //disable feed requests by default
        const hasDisableBtn = this.props.AdminUser?.blockRules?.ShowDisableFeedBtn ?? true;

        return hasDisableBtn && this.cs[CSSettingsItems.disableFeedRequests];
    }

    startWorker() {
        const { isChatRole, isCallRole, isDutyRole, isFeedRequestsDisabled } = this.getReasons();
        (isChatRole || isCallRole || isDutyRole) && !isFeedRequestsDisabled && this.initWorker();

        //@ts-ignore
        if (window?.SharedWorker) {
            new BackgroundTabs({
                onSlave: this.onSlave.bind(this),
                onMaster: this.onMaster.bind(this),
            });
        } else {
            this.onMaster();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<ChatNotifyProps>, nextState: Readonly<any>, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps);
    }

    isContainChatRoles(attribute) {
        const roles = this.props.AdminUser
            && this.props.AdminUser.activeRoles
            && Object.entries(this.props.AdminUser.activeRoles) || [];

        return roles.find(role => {
            return role[1].roleGrouppingTags && role[1].roleGrouppingTags.toUpperCase().includes(attribute) || false;
        });
    }

    isContainInAllRoles(attribute) {
        return this.props?.AdminUser?.rolesRaw?.find((role: typeof MY_ROLES_LIST) => {
            return role?.role_description?.role_groupping_tags
                .toString()?.toUpperCase().includes(attribute.toUpperCase());
        });
    }

    isFeedRequestsDisabled() {
        const isChats = this.props.location.pathname === CHATS_PATH;

        return this.checkForDisableFeed() && !isChats;
    }

    getReasons(): { isChatRole; isCallRole; isDutyRole; isFeedRequestsDisabled } {
        const isChatRole = this.isContainChatRoles(CHAT_ROLES_TAG);
        const isCallRole = this.isContainChatRoles(CALL_ROLES_TAG);
        const isDutyRole = this.isContainInAllRoles(DUTY_ROLES_TAG);
        const isFeedRequestsDisabled = this.isFeedRequestsDisabled();

        return { isChatRole, isCallRole, isDutyRole, isFeedRequestsDisabled };
    }

    initChecking(chat, call, duty) {
        this.worker.postMessage({
            type: CHATS_COMMAND.INIT_CHECKING_QUEUE,
            ls: new LS().getAllData(),
            options: {
                chat,
                call,
                duty,
            },
        });
    }

    finishChecking() {
        this.worker.postMessage({ type: CHATS_COMMAND.FINISH_CHECKING_QUEUE });
    }

    initWorker() {
        this.worker.onmessage = (event: any) => {
            if (event.data.type === CHATS_COMMAND.UPDATE_WORKER_IS_READY) {
                const props = this.props.AdminUser;
                this.worker.postMessage({
                    type: CHATS_COMMAND.START_CHECKING_QUEUE,
                    props,
                    options: this.getReasons(),
                });
            }

            if (event.data.type === CHATS_COMMAND.SET_CHATS_LOADING) {
                this.props.setChatsLoading({ chatType: event.data.chatType, state: event.data.state });
            }

            if (event.data.type === CHATS_COMMAND.SET_CHATS) {
                const payload = event.data.payload;
                this.props.setChats(payload, null);
                this.props.setChatsLoading({ chatType: payload?.type, state: false });

                if (payload?.response?.menu?.[1]) {
                    payload?.response?.menu[0]?.forEach(lineItem => {
                        const check = this.favLines.check(payload.type, lineItem);
                        if (check) {
                            this.favLines.updateChatsCount(check.group, check.line?.name, check.targetChats);
                            const line = check.line?.display_name?.toUpperCase() || check.line.name;
                            this.chatNotification.notify(line, FAV_LINES_TYPES.chats);
                        }
                    });
                }
            }
        };
    }

    openChats() {
        location.href = '#/chats';
    }

    isState(state) {
        return isFeedState.call(this, state);
    }

    render() {
        const unreadCount = this.props.myChats?.total_unread;

        return <div className={`${style.notify_chat} ${style.active}`}>
            <ChatIcon onClick={this.openChats.bind(this)}/>
            <div className={`${style.unread_count} ${!unreadCount ? style.hide : ''} ${unreadCount > MORE_COUNT ? style.more : ''}`}>
                {unreadCount > MORE_COUNT ? MORE_COUNT + '+' : unreadCount}
            </div>
            {
                this.isState(DUTY_STATE.SUPPOSEDLY_AFK)
                    ? <AbsenceModal tags={this.props.myChats?.status_tags}/>
                    : this.isState(DUTY_STATE.ENDED)
                        ? <DutyEnter tags={this.props.myChats?.status_tags}/>
                        : this.isState(DUTY_STATE.BREAK)
                            ? <FromBreakConfirm tags={this.props.myChats?.status_tags}/>
                            : null
            }
        </div>;
    }
}
