import React from 'react';

import { ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { getRenderedBBCode } from '../../../ui/BBCodeRenderer';
import FormatDate from '../../../ui/FormatDate';
import { ErrorSource, logError } from '../../Content/initErrorCounter';
import * as style from '../index.css';

interface IChatMessageItemProps {
    item: any;
    users: any;
}

export enum OUTGOING_MESSAGE_TYPES {
    plaintext = 'plaintext',
    longterm_target_point = 'longterm_target_point',
    note = 'note',
}

export class ChatMessageItem extends React.Component<IChatMessageItemProps, any> {
    state = {
        error: null,
    };

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);

        this.setState({ error });
    }

    getMessage(): any {
        const props = {
            item: this.props.item,
            users: this.props.users,
            error: this.state.error,
        };

        switch (this.props.item.type) {
        case OUTGOING_MESSAGE_TYPES.longterm_target_point:
            return <LongtermTargetPointMessage {...props}/>;
        default:
            return <PlaintextMessage {...props}/>;
        }
    }

    render() {
        const message = this.getMessage();

        return <div className={style.comment_item}>
            {message}
        </div>;
    }
}

interface IMessageProps {
    item: any;
    users: any[];
    error: Error | null;
}

const PlaintextMessage = React.memo((props: IMessageProps) => {
    const author = props.users?.[props.item?.author];

    return <>
        <div className={style.comment_item_date}>
            <FormatDate value={props.item.timestamp * ONE_SECOND} withSecond/>
        </div>
        <div className={style.comment_item_author}>
            {author && UserInfoHandler.getPrintName.call(author) || props.item.author}
        </div>

        {props.error
            ? <div className={style.warning}>
                <div className={style.warning_text}><i>Возможно, поехала верстка!</i></div>
                {props.item.text}
            </div>
            : <div className={style.comment_item_text}>{getRenderedBBCode(props.item.text)}</div>
        }
    </>;
});

const LongtermTargetPointMessage = React.memo((props: IMessageProps) => {
    return <>
        <PlaintextMessage {...props}/>
        <img src={props.item?.src} alt={'Точка подгона авто'}/>
    </>;
});
