import React, { useState } from 'react';
import VirtualList from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_DAY, ONE_KILOMETER, ONE_SECOND } from '../../../../constants';
import { CarInfoHandler } from '../../../../models/car';
import { CancelButton, SaveButton } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import * as coreStyle from '../../../../ui/index.css';
import { Link } from '../../../../ui/Link';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import CarNumber from '../../../CarNumber';
import { FormConstructor } from '../../../FormConstructor';
import { initMapApi } from '../../../MainMap/utils';
import Spin from '../../../Spin';
import * as style from '../../index.css';
import { CITY_TAGS, NEW_AUTO_TAG } from '../constants';
import { EVOLUTION_SCHEMA } from './evolutionSchema';
import { IMapStateToProps } from './index';
import { CHAT_REQUESTS, REQUESTS } from './requests';

interface IEvolutionModalProps extends IMapStateToProps {
    openEvolutionModal: (isOpen?: boolean) => void;
    longtermData: any;
    item: any;
}

interface IEvolutionModalState {
    carId: string;
    evolution: string;
    isLoading: boolean;
    error: Error | null;

    isCarsLoading: boolean;
    carsError: Error | null;
    filteredCars: any[];
    needAdditionalFilter: boolean;
}

const LIST_HEIGHT = 320;
const LIST_ITEM_HEIGHT = 30;
const FREE_USAGE_KEY = 'free';
const MILEAGE_KEY = 'mileage';
const AVAILABLE_STATUS = 'available';
const HOLD_FOR_LT_STATUS = 'hold_for_long_term';
const PROCESS_DAYS = 2;

export class LongtermModal extends React.Component<IEvolutionModalProps, IEvolutionModalState> {
    state: IEvolutionModalState = {
        carId: '',
        evolution: '',
        isLoading: false,
        error: null,

        isCarsLoading: false,
        carsError: null,
        filteredCars: [],
        needAdditionalFilter: true,
    };
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    ymaps;

    componentDidMount() {
        this.getCars();
        this.initMap();
    }

    shouldComponentUpdate(
        nextProps: Readonly<IEvolutionModalProps>,
        nextState: Readonly<IEvolutionModalState>,
    ): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    getCars() {
        this.setState({ isCarsLoading: true, carsError: null }, () => {
            const action_id = this.props.longtermData?.Offer?.BehaviourConstructorId ?? '';

            this.request.exec(REQUESTS.GET_ACTION, {
                queryParams: { action_id },
            })
                .then(response => {
                    const actionInfo = response?.report?.[0]?.action_meta ?? {};
                    const responseCarTags = actionInfo.car_tags_filter;
                    const tags = actionInfo.offer_tags;
                    const cityTag = tags.find(el => el.includes('city'));

                    if (responseCarTags) {
                        this.setState({ needAdditionalFilter: false });
                    }

                    return responseCarTags ? responseCarTags : this.props.carTags + CITY_TAGS[cityTag]?.tag;
                })
                .then(tags_filter => {
                    this.request.exec(REQUESTS.GET_CAR_LIST, {
                        queryParams: {
                            sensors: MILEAGE_KEY,
                            tags_filter: encodeURIComponent(tags_filter),
                        },
                    })
                        .then(response => {
                            this.setState({
                                isCarsLoading: false,
                                filteredCars: this.filterCars(response),
                            });
                        })
                        .catch(carsError => this.setState({
                            carsError,
                            isCarsLoading: false,
                        }));
                })
                .catch(carsError => this.setState({
                    carsError,
                    isCarsLoading: false,
                }));
        });
    }

    filterCars(data) {
        const model = this.props?.longtermData?.Offer?.ObjectModel;
        const carsWithoutMileage: any = [];
        const cih = CarInfoHandler;

        const filteredCars = model && data?.cars
            ?.filter(car => {
                const isFree = car.usage === FREE_USAGE_KEY;
                const correctStatus = car.status === AVAILABLE_STATUS || car.status === HOLD_FOR_LT_STATUS;
                const correctModel = car.model_id === model;

                const filter = this.state.needAdditionalFilter
                    ? isFree && correctStatus && correctModel
                    : isFree && correctStatus;

                if (filter) {
                    if (!cih.getMileage.call(car)) {
                        carsWithoutMileage.push(car);

                        return;
                    }

                    return car;

                }
            })
            ?.sort((car1, car2) => {
                return cih.getMileage.call(car1) - cih.getMileage.call(car2);
            });

        return [...filteredCars, ...carsWithoutMileage];
    }

    getOptions() {
        return this.getTagEvolution().map(el => {
            return { text: el.to, value: el.to };
        }) ?? [];
    }

    getTagEvolution() {
        return this.props.evolutions?.[NEW_AUTO_TAG] || [];
    }

    getTagId() {
        return this.props.item?.tag_id || this.props.item?.tag_data?.tag_id;
    }

    onSave() {
        this.setState({ isLoading: true, error: null }, () => {
            const tag = this.getTagEvolution()[0]?.to;
            tag && this.request.exec(REQUESTS.TAG_EVOLVE, {
                queryParams: {
                    tag_id: this.getTagId(),
                },
                body: {
                    tag,
                    topic_link: this.props.item?.topic_link,
                    CarId: this.state.carId,
                },
            })
                .then(() => {
                    this.setState({ isLoading: false });
                    this.props.openEvolutionModal();
                })
                .catch(error => {
                    const busyCarInfo = 'car tag old_state_riding has performer';
                    const busyCarMark = error.data?.error_details?.special_info?.session_info['LongTermOfferHolderTag::OnBeforeEvolve']?.includes(busyCarInfo);
                    if (busyCarMark) {
                        this.setState({
                            error: {
                                name: 'Busy car',
                                message: 'Машина уже забронирована, попробуйте найти другую',
                            },
                            isLoading: false,
                        });
                    } else {
                        this.setState({ error, isLoading: false });
                    }
                });
        });
    }

    onSchemaChange(data) {
        this.setState({ carId: data?.carId || '', evolution: data?.evolution || '' });
    }

    choseCar(item) {
        this.setState({ carId: item?.id });
    }

    initMap() {
        // @ts-ignore
        const ymaps = window.ymaps;
        if (!ymaps) {
            const mapScript = initMapApi();
            mapScript.addEventListener('load', () => {
                // @ts-ignore
                this.ymaps = window.ymaps;
            });
        } else {
            this.ymaps = ymaps;
        }
    }

    render() {
        const { carId, evolution, error, isLoading, filteredCars, isCarsLoading, carsError } = this.state;
        const options = this.getOptions();
        EVOLUTION_SCHEMA.evolution.variants = options;
        const initialData = {
            evolution: options[0]?.value,
            carId: carId || this.props.longtermData?.CarId,
        };

        const DeliveryLocation = this.props.longtermData?.Offer?.LongTermOffer?.DeliveryLocation;
        const location = [DeliveryLocation?.YHP, DeliveryLocation?.XHP];
        const longtermStart = this.props.longtermData?.Offer?.LongTermOffer?.Since / ONE_SECOND;
        const processTime = (new Date()).setHours(0, 0, 0, 0) + ONE_DAY * PROCESS_DAYS;

        return <Window onClose={this.props.openEvolutionModal.bind(this)}
                       title={'Привязать ID машины'}
                       error={error || carsError}
                       className={style.longterm_modal}>
            {longtermStart > processTime
                ? <div className={style.longterm_warning}>
                    Дата начала аренды больше завтрашнего дня. Эту заявку нужно отложить
                </div>
                : null
            }

            <FormConstructor schema={EVOLUTION_SCHEMA}
                             onChange={this.onSchemaChange.bind(this)}
                             initialData={initialData}/>

            {isCarsLoading
                ? <Spin/>
                : filteredCars?.length
                    ? <div className={style.table}>
                        <span className={style.table_title}>Пробег</span>
                        <span className={style.table_title}>Расстояние (км)</span>
                        <VirtualList height={LIST_HEIGHT}
                                     itemCount={filteredCars.length}
                                     itemSize={LIST_ITEM_HEIGHT}
                                     renderItem={({ index, style }) => {
                                         return <CarInfoItem key={index + 1}
                                                             style={style}
                                                             item={filteredCars[index]}
                                                             location={location}
                                                             ymaps={this.ymaps}
                                                             choseCar={this.choseCar.bind(this)}/>;
                                     }
                                     }/>
                    </div>
                    : <div className={style.table}>Свободных машин по такому запросу нет:( </div>
            }

            <div className={coreStyle.button_container}>
                <CancelButton onClick={this.props.openEvolutionModal.bind(this)} isLoading={isLoading}/>
                <SaveButton onClick={this.onSave.bind(this)} disabled={!carId || !evolution}/>
            </div>
        </Window>;
    }
}

interface ICarInfoItemProps {
    item: any;
    choseCar: (item: any) => void;
    style: any;
    key: number;
    location: number[];
    ymaps: any;
}

const CarInfoItem = React.memo((props: ICarInfoItemProps) => {
    const mileage = CarInfoHandler.getMileage.call(props.item);
    const carLocation = [props.item.location.lat, props.item.location.lon];
    const [distance, setDistance] = useState(0);
    getDistanceInMeters(props.location, carLocation);

    function getDistanceInMeters(loc1, loc2) {
        props.ymaps.ready(() => {
            setDistance(props.ymaps.coordSystem.geo.getDistance(loc1, loc2));
        });
    }

    return props.item?.number ?
        <table style={props.style}>
            <tbody>
                <tr>
                    <td className={style.numbers_data}>{mileage || EMPTY_DATA}</td>
                    <td className={style.numbers_data}>
                        {distance ? (distance / ONE_KILOMETER).toFixed(1) : EMPTY_DATA}
                    </td>
                    <td><CarNumber value={props.item?.number}/></td>
                    <td><Link onClick={props.choseCar.bind(null, props.item)}>Выбрать</Link></td>
                    <td><Link href={`#/cars/${props.item.id}/acceptance_photos`} target={'_blank'}>Перейти</Link></td>
                </tr>
            </tbody>
        </table>
        : null;
});
