import React, { ReactElement } from 'react';

import { GetClassification, IGetClassificationExport } from '../../../../decorators/GetClassification';
import { ITagsWithPropsExport, UserTags } from '../../../../decorators/UserTags';
import { TIMERS } from '../../../constants';
import { RichEditor } from '../../../ui/RichEditor';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { _CHATS, NChats } from '../../ChatsCommon/types';
import * as style from '../index.css';
import { requestConfigs, REQUESTS } from '../request';
import { ChatWindowContent } from './ChatWindowContent';
import ChatWindowHeader from './ChatWindowHeader';
import { DAMAGES_PHOTOS_TAGS, LEASING_SUPPORT_TAG } from './constants';
import LongtermModal from './LongtermModal';
import ISelectedChatData = NChats.ISelectedChatData;

interface IChatWindowProps extends ITagsWithPropsExport, IGetClassificationExport {
    topic_link: string | null;
    user_id: string | null;
    tag_id: string | null;
    menu: string | null;

    tasks: any;
    selectedChatItem: any;
    isEditorClosedByDefault: boolean | undefined;
    dispatcherLongtermTags: string[];
}

interface IChatWindowState {
    commentsIsLoading: boolean;
    selectedChatData: ISelectedChatData;
    longtermModalOpen: boolean;
    users: any;
}

@UserTags()
@GetClassification()
export class ChatWindow extends React.Component<IChatWindowProps, IChatWindowState> {
    state: IChatWindowState = {
        commentsIsLoading: true,
        selectedChatData: {} as ISelectedChatData,
        longtermModalOpen: false,
        users: {},
    };
    richEditor;
    fastAnswers = [
        { name: 'test', text: 'test text', tags: ['tag1', 'tag2'] },
        {
            name: 'link',
            text: 'а вот, например, [url=#/settings/error-log]ссылка[/url] на ошибки в [color=#0ff]адмике[/color]',
            tags: ['link', 'href'],
        },
    ];
    request = new Request2({ requestConfigs });
    timeout;

    constructor(props) {
        super(props);
        this.richEditor = React.createRef();
    }

    componentDidMount(): void {
        this.getComments(true);
    }

    shouldComponentUpdate(
        nextProps: Readonly<IChatWindowProps>,
        nextState: Readonly<IChatWindowState>,
        nextContext: any,
    ): boolean {
        return (!isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState));
    }

    componentDidUpdate(prevProps: Readonly<IChatWindowProps>, prevState: Readonly<any>, snapshot?: any): void {
        if (!isObjectEqual(this.props, prevProps)) {
            const originator = this.props?.selectedChatItem?.originator;
            const topic_link = this.props?.selectedChatItem?.topic_link;

            if (originator !== prevProps?.selectedChatItem?.originator
                || topic_link !== prevProps?.selectedChatItem?.topic_link
            ) {
                this.getComments(true);
                this.props?.getTags?.({ object_id: originator });
                this.props.tag_id && this.props.getClassification?.(this.props.tag_id);
            }
        }
    }

    componentWillUnmount() {
        clearTimeout(this.timeout);
        this.request.abort();
    }

    checkIfLongterm() {
        const { dispatcherLongtermTags, selectedChatItem } = this.props;

        return dispatcherLongtermTags?.includes(selectedChatItem?.tag_data?.tag);
    }

    sortChatMessages() {
        const expandedMessages: any[] = this.state.selectedChatData?.messages?.slice();
        expandedMessages?.sort((m1, m2) => m1.timestamp - m2.timestamp);

        return expandedMessages;
    }

    getLongtermData() {
        const { selectedChatItem, tags } = this.props;
        const linked_id = selectedChatItem.linked_tag_id ?? selectedChatItem.tag_data?.linked_tag_id;

        return tags?.find(el => el.tag_id === linked_id);
    }

    disableRichEditor(state: boolean) {
        this.richEditor && this.richEditor.current && this.richEditor.current.disable(state);
    }

    setStatusText(statusText: ReactElement | string, options?: any) {
        this.richEditor && this.richEditor.current && this.richEditor.current.setStatusText(statusText, options);
    }

    setReachEditText(text: string) {
        this.richEditor && this.richEditor.current && this.richEditor.current.setText(text);
    }

    onSendText(data) {
        this.setStatusText('отправка комментария');
        this.disableRichEditor(true);

        this.request.exec(REQUESTS.SEND_MESSAGE, {
            queryParams: {
                user_id: this.props.user_id,
                chat_id: this.props.topic_link,
            },
            body: {
                'message': data,
                'traits': ['staff_only'],
            },
        })
            .then(() => {
                this.setStatusText('');
                this.disableRichEditor(false);
                this.setReachEditText('');
                this.getComments(false);
            })
            .catch(errorData => {
                this.setStatusText('', { error: true, errorData });
                this.disableRichEditor(false);
            });
    }

    getComments(commentsIsLoading: boolean) {
        if (this.props.menu === _CHATS.ARCHIVE) {
            this.disableRichEditor(true);
        }

        clearTimeout(this.timeout);

        const selectedChatData = commentsIsLoading ? {} as ISelectedChatData : this.state.selectedChatData;
        this.props.topic_link && this.setState({
            commentsIsLoading,
            selectedChatData,
        }, () => {
            this.request.exec(REQUESTS.GET_COMMENTS, {
                queryParams: {
                    user_id: this.props.user_id,
                    chat_id: this.props.topic_link,
                },
            })
                .then(selectedChatData => {
                    const users = {};
                    selectedChatData?.users?.forEach(el => users[el.id] = el);

                    this.setState({
                        selectedChatData,
                        users,
                        commentsIsLoading: false,
                    });
                    this.repeat();
                })
                .catch(() => {
                    this.setState({
                        commentsIsLoading: false,
                    });
                });
        });
    }

    repeat() {
        clearTimeout(this.timeout);
        this.timeout = setTimeout(this.getComments.bind(this, false), TIMERS.OUTGOING_CHATS);
    }

    openEvolutionModal(longtermModalOpen) {
        this.setState({ longtermModalOpen });
    }

    getDamagesSessionId() {
        const currentTagName = this.props.selectedChatItem?.tag_data?.tag || this.props.selectedChatItem?.tag;
        const isDamageTask = DAMAGES_PHOTOS_TAGS.some(el => el === currentTagName);

        return isDamageTask
            ? this.props.selectedChatItem?.tag_data?.comment || this.props.selectedChatItem?.tag_data?.session_id
            : null;
    }

    getLeasingSupportData() {
        const { selectedChatItem } = this.props;
        const currentTagName = selectedChatItem?.tag_data?.tag || selectedChatItem?.tag;
        try {
            return currentTagName === LEASING_SUPPORT_TAG
                ? JSON.parse(selectedChatItem?.tag_data?.comment)
                : null;
        } catch {
            return null;
        }
    }

    render() {
        const tagData = this.props.selectedChatItem?.tag_data ?? {};
        const chatDisplayName = tagData.tag && this.props.tasks?.tags?.[tagData.tag]?.display_name;
        const title = chatDisplayName || this.props.selectedChatItem?.name || this.props.selectedChatItem?.title;
        const longtermData = this.checkIfLongterm() ? this.getLongtermData() : null;
        const damages_session_id = this.getDamagesSessionId();
        const leasingSupportData = this.getLeasingSupportData();
        const messages = this.sortChatMessages();
        const classificationParams = {
            chat_id: tagData.topic_link,
            user_id: this.props.user_id,
            tag_id: this.props.tag_id,
        };

        return this.props.topic_link && <>
            <ChatWindowHeader classificationData={this.props.classificationData}
                              classificationParams={classificationParams}
                              getClassification={this.props.getClassification?.bind(this)}
                              title={title}
                              commentsIsLoading={this.state.commentsIsLoading}
                              selectedChatItem={this.props.selectedChatItem}/>

            <div className={style.comments}>
                <ChatWindowContent selectedChatItem={this.props.selectedChatItem}
                                   users={this.state.users}
                                   messages={messages}
                                   longtermData={longtermData}
                                   damagesSessionId={damages_session_id}
                                   leasingSupportData={leasingSupportData}
                                   openEvolutionModal={this.openEvolutionModal.bind(this, true)}
                                   classification_comment={this.props.classificationData}/>
            </div>

            <RichEditor initData={''}
                        fastAnswers={this.fastAnswers}
                        ref={this.richEditor}
                        className={`${style.editor}`}
                        tag_id={this.props.tag_id}
                        isClosedByDefault={this.props.isEditorClosedByDefault}
                        onSendText={this.onSendText.bind(this)}/>

            {this.state.longtermModalOpen
                ? <LongtermModal openEvolutionModal={this.openEvolutionModal.bind(this, false)}
                                 longtermData={longtermData}
                                 item={this.props.selectedChatItem}/>
                : null
            }
        </>;
    }
}
